<?php

namespace Controllers;

use Core\View;
use Core\Controller;
use Helpers\Auth;
use Helpers\Url;
use Models\Pms;
use Models\Server;
use Models\User;
use ZMQ;
use ZMQContext;
use ZMQPoll;
use ZMQSocket;

/*
 * Copyright 2014
 * ParaBlu Systems Private Limited
 * All Rights Reserved
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of ParaBlu Systems Private Limited and its suppliers, if any.
 * The intellectual and technical concepts contained herein are proprietary
 * to ParaBlu Systems Private Limited and its suppliers and may be covered by
 * Indian, US and Foreign Patents, patents in process, and are protected by
 * trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from ParaBlu Systems Private Limited.
 *
 * Login controller
 *
 * @author Prashanth Kumar B - prashanth.kumar@parablu.com
 * @version 2.0
 * @date June 10, 2015
 *
 */

class ServerController extends Controller{

    /**
     * Below variable contains the object reference of a model
     */

    protected $_model;

    /**
     * Call the parent construct
     */
    public function __construct(){
        parent::__construct();
        $this->auth->loginCheck();
        $this->language->load('server');
        $this->_model = new Server();
    }

    public function listOfServers(){

        $data = array(
            'title' => $this->language->get('Server'),
            'crumb1' => $this->language->get('List Of Servers'),
            'active' => 'servers',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('server/servers', $data);
        View::renderTemplate('footer', $data);
    }

    public function getAllDevices(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST)){

            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token'),
                'skipValue' => $_POST['skipValue']
            );

            if($_POST['deviceName'] != 'all'){
                $headers['deviceNames'] = $_POST['deviceName'];
                $result = $this->_model->searchInDevices($headers);
            }else{
                $result = $this->_model->getAllDevicesWithUserName($headers, $this->session->get('userName'));
            }

            switch ($result['status']) {
                case (200):

                    $devices = [];

                    if(!empty($result['body'])){
                        foreach($result['body']->{'device-info'} as $value){
                            $device = array(
                                'profilePic' => Url::profilePic((string)$value->{'userName'}),
                                'decoupled' => (string)$value->{'decoupled'},
                                'createdTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'device-created-timestamp'}) / 1000)),
                                'modifiedTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'modified-timestamp'}) / 1000)),
                                'modifiedTimeStampUnix' => (int)$value->{'modified-timestamp'},
                                'deviceName' => (string)$value->{'device-name'},
                                'osType' => (string)$value->{'device-os-type'},
                                'icon' => $this->getOsIcon($value->{'device-os-type'}),
                                'version' => (string)$value->{'device-os-version'},
                                'tempPath' => (string)$value->{'device-temp-path'},
                                'deviceUUID' => (string)$value->{'device-uuid'},
                                'blocked' => (string)$value->{'is-blocked'},
                                'userName' => (string)$value->{'userName'},
                            );
                            $devices = array_merge(array($device), $devices);
                        }
                    }

                    $response = array(
                        'result' => 'success',
                        'devices' => $devices
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function addAServer(){

        $data = array(
            'title' => $this->language->get('Server'),
            'crumb1' => $this->language->get('Add New Server'),
            'active' => 'servers',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('server/addNewServer', $data);
        View::renderTemplate('footer', $data);
    }

    public function getAllServers(){
        if(!empty($_GET)){

            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token'),
                'skipValue' => $_GET['skipValue']
            );

            if(!empty($_GET['q']) && $_GET['q'] != 'all'){
                $headers['deviceNames'] = $_GET['q'];
                $result = $this->_model->searchInDevices($headers);
            }else{
                $result = $this->_model->getAllDevices($headers);
            }

            switch ($result['status']) {
                case (200):

                    $devices = [];
                    $devices = array_merge(array(array('deviceName' => 'All', 'deviceUUID' => 'all', 'profilePic' => Url::profilePic('default'))), $devices);
                    if(!empty($result['body'])){
                        foreach($result['body']->{'device-info'} as $value){
                            $device = array(
                                'profilePic' => Url::profilePic((string)$value->{'userName'}),
                                'decoupled' => (string)$value->{'decoupled'},
                                'createdTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'device-created-timestamp'}) / 1000)),
                                'modifiedTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'modified-timestamp'}) / 1000)),
                                'modifiedTimeStampUnix' => (int)$value->{'modified-timestamp'},
                                'deviceName' => (string)$value->{'device-name'},
                                'osType' => (string)$value->{'device-os-type'},
                                'icon' => (strpos((string)$value->{'device-os-type'},'Win') !== false ? 'windows' : 'apple'),
                                'version' => (string)$value->{'device-os-version'},
                                'tempPath' => (string)$value->{'device-temp-path'},
                                'deviceUUID' => (string)$value->{'device-uuid'},
                                'blocked' => (string)$value->{'is-blocked'},
                                'userName' => (string)$value->{'userName'},
                            );
                            $devices = array_merge(array($device), $devices);
                        }
                    }

                    $response = array(
                        'result' => 'success',
                        'devices' => $devices
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function listOfServerJobs(){

        $data = array(
            'title' => $this->language->get('Server'),
            'crumb1' => $this->language->get('List Of Server Jobs'),
            'active' => 'jobs',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('server/listOfServerJobs', $data);
        View::renderTemplate('footer', $data);
    }


    public function getListOfServerJobs(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST)){
            $headers = array(
                'deviceUUID'     => !empty($_POST['deviceUUID']) ? $_POST['deviceUUID'] : 'all',
                'status'         => !empty($_POST['status']) ? $_POST['status'] : 'running',
                'skipValue'      => $_POST['skipValue'],
                'token'          => $this->session->get('token')
            );
            $this->log('Sent Data'. json_encode($headers), 'info');
            $result = $this->_model->getListOfServerJobs($headers);
            $this->log('Received Data'. json_encode($result), 'info');
            switch ($result['status']) {
                case (200):
                    $result = array(
                        'result' => 'success',
                        'serverJobs' => $result['body']->{'deviceBackupJobElementList'}
                    );
                    echo json_encode($result);
                    exit();
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    public function initiateBackup(){

        if(empty($_POST['serverHostName'])){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
            exit;
        }

        $data = array(
            'action'            => 'BACKUP',
            'containerName'     => $_POST['selectedContainerName'],
            'type'              => $_POST['type']
        );

        $context = new ZMQContext();
        $client = $this->client_socket($context, $_POST['serverHostName']);

        $sequence = 0;
        $retries_left = 3;
        $read = $write = array();

        while ($retries_left) {
            $client->send(json_encode($data));
            $expect_reply = true;
            while ($expect_reply) {

                $poll = new ZMQPoll();
                $poll->add($client, ZMQ::POLL_IN);
                $events = $poll->poll($read, $write, -1);
                --$retries_left;
                if ($events > 0) {

                    $result = $client->recv();

                    switch ($result) {
                        case (200):

                            $response = array(
                                'result' => 'success',
                                'message' => $this->language->get('backupInitiated')
                            );
                            echo json_encode($response);
                            exit;

                        case (421):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('421')
                            );
                            echo json_encode($result);
                            exit();
                        case (500):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('500')
                            );
                            echo json_encode($result);
                            exit();
                        case (0):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('0')
                            );
                            echo json_encode($result);
                            exit();
                        default:
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('default')
                            );
                            echo json_encode($result);
                            exit();
                    }
                } elseif ($retries_left == 0) {
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    break;
                } else {
                    $this->log('Retrying to connect to client:'.$_POST['serverHostName'], 'info');
                    $client = $this->client_socket($context, $_POST['serverHostName']);
                    $client->send($sequence);
                }
            }
        }
    }

    public function attachContainer($deviceName, $deviceUUID){
        $headers = array(
            'Content-type' => 'application/json',
            'Accept' => 'application/json',
            'token'     => $this->session->get('token')
        );
        $userModel = new User();
        $groupResult = $userModel->getUserGroups($headers);
        $groups = !empty($groupResult['body']) ? json_decode($groupResult['body']) : '';

        $pmsModel = new Pms();
        $result = $pmsModel->getAllPolicies(array('token'=>$this->session->get('token')));
        $policies = [];
        if(!empty((array)$result['body']->{'backupPolicyElementList'})){
            foreach((array)$result['body']->{'backupPolicyElementList'} as $value){
                $policy = array('name' => $value->{'policygroupName'});
                $policies = array_merge(array($policy), $policies);
            }
        }


        $data = array(
            'title' => $this->language->get('Server'),
            'crumb1' => $this->language->get('Container'),
            'active' => 'servers',
            'policies' => $policies,
            'groups' => $groups->stringElementList,
            'deviceUUID' => htmlspecialchars($deviceUUID),
            'deviceName' => $deviceName,
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('server/attachContainer', $data);
        View::renderTemplate('footer', $data);
    }

    public function saveContainer(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST)){
            $headers = array(
                'Content-type' => 'application/json',
                'Accept' => 'application/json',
                'token'     => $this->session->get('token')
            );

            $data = array(
                'containerName' => $_POST['containerName'],
                'containerType' => $_POST['containerType'],
                'containerLocation' => $_POST['fileSource'],
                'username' => $_POST['dbUserName'],
                'password' => $_POST['dbPassword'],
                'port' => $_POST['dbPort'],
                'deviceUUID' => $_POST['deviceUUID'],
                'backupPolicyName' => $_POST['containerPolicy'],
                'groupName' => $_POST['userGroup']
            );

            $result = $this->_model->saveContainer($headers, $data);

            switch ($result['status']) {
                case (200):
                    $result = array(
                        'result' => 'success',
                        'msg' => $result
                    );
                    echo json_encode($result);
                    exit();
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (601):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('601')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    public function editSaveContainer(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST)){
            $headers = array(
                'Content-type' => 'application/json',
                'Accept' => 'application/json',
                'token'     => $this->session->get('token')
            );

            $data = array(
                'id'                => $_POST['id'],
                'containerName'     => $_POST['containerName'],
                'containerType'     => $_POST['containerType'],
                'containerLocation' => $_POST['containerLocation'],
                'username'          => $_POST['username'],
                'password'          => $_POST['password'],
                'port'              => $_POST['port'],
                'deviceUUID'        => $_POST['deviceUUID'],
                'backupPolicyName'  => $_POST['backupPolicyName'],
                'groupName'         => $_POST['groupName']
            );
            $this->log("Data". json_encode($data), 'info');
            $result = $this->_model->saveEditedContainer($headers, $data);
            $this->log("result". json_encode($result), 'info');
            switch ($result['status']) {
                case (200):
                    $result = array(
                        'result' => 'success',
                        'msg' => $result
                    );
                    echo json_encode($result);
                    exit();
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (601):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('601')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    public function removeContainer(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST)){
            $headers = array(
                'Content-type' => 'application/json',
                'Accept' => 'application/json',
                'userName'     => $this->session->get('userName')
            );

            $data = array(
                'id' => $_POST['id'],
                'containerName' => $_POST['containerName'],
                'containerType' => $_POST['containerType'],
                'containerLocation' => $_POST['fileSource'],
                'username' => $_POST['dbUserName'],
                'password' => $_POST['dbPassword'],
                'port' => $_POST['dbPort'],
                'deviceUUID' => $_POST['deviceUUID'],
                'backupPolicyName' => $_POST['containerPolicy'],
                'groupName' => $_POST['userGroup']
            );

            $result = $this->_model->removeContainer($headers, $data);

            switch ($result['status']) {
                case (200):
                    $result = array(
                        'result' => 'success',
                        'msg' => $result
                    );
                    echo json_encode($result);
                    exit();
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    public function getAllContainers(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST)){
            $headers = array(
                'Content-type' => 'application/json',
                'Accept' => 'application/json',
                'token'     => $this->session->get('token'),
                'deviceUUID'     => $_POST['deviceUUID'],
                'groupName'     => '',
                'userName'     => $this->session->get('userName')
            );

            $result = $this->_model->getAllContainers($headers);

            switch ($result['status']) {
                case (200):
                    $result = array(
                        'result' => 'success',
                        'data' => $result['body']->{'editDeviceContainerElementList'}
                    );
                    echo json_encode($result);
                    exit();
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    public function initiateRefreshPolicy(){
        if(empty($_POST['serverHostName'])){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
            exit;
        }
        $data = array(
            'action' => 'REFRESHPOLICY'
        );

        $context = new ZMQContext();
        $client = $this->client_socket($context, $_POST['serverHostName']);

        $sequence = 0;
        $retries_left = 3;
        $read = $write = array();

        while ($retries_left) {
            $client->send(json_encode($data));
            $expect_reply = true;
            while ($expect_reply) {

                $poll = new ZMQPoll();
                $poll->add($client, ZMQ::POLL_IN);
                $events = $poll->poll($read, $write, -1);
                --$retries_left;
                if ($events > 0) {

                    $result = $client->recv();

                    switch ($result) {
                        case (200):

                            $response = array(
                                'result' => 'success',
                                'message' => $this->language->get('refreshPolicy')
                            );
                            echo json_encode($response);
                            exit;

                        case (421):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('421')
                            );
                            echo json_encode($result);
                            exit();
                        case (500):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('500')
                            );
                            echo json_encode($result);
                            exit();
                        case (0):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('0')
                            );
                            echo json_encode($result);
                            exit();
                        default:
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('default')
                            );
                            echo json_encode($result);
                            exit();
                    }
                } elseif ($retries_left == 0) {
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    break;
                } else {
                    $this->log('Retrying to connect to client:'.$_POST['serverHostName'], 'info');
                    $client = $this->client_socket($context, $_POST['serverHostName']);
                    $client->send($sequence);
                }
            }
        }
    }

    public function decoupleAgent(){
        if(empty($_POST['serverHostName'])) {
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
            exit;
        }
        $data = array(
            'action' => 'DECOUPLE'
        );

        $context = new ZMQContext();
        $client = $this->client_socket($context, $_POST['serverHostName']);

        $sequence = 0;
        $retries_left = 3;
        $read = $write = array();

        while ($retries_left) {
            $client->send(json_encode($data));
            $expect_reply = true;
            while ($expect_reply) {

                $poll = new ZMQPoll();
                $poll->add($client, ZMQ::POLL_IN);
                $events = $poll->poll($read, $write, -1);
                --$retries_left;
                if ($events > 0) {

                    $result = $client->recv();

                    switch ($result) {
                        case (200):

                            $response = array(
                                'result' => 'success',
                                'message' => $this->language->get('decouple')
                            );
                            echo json_encode($response);
                            exit;

                        case (421):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('421')
                            );
                            echo json_encode($result);
                            exit();
                        case (500):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('500')
                            );
                            echo json_encode($result);
                            exit();
                        case (0):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('0')
                            );
                            echo json_encode($result);
                            exit();
                        default:
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('default')
                            );
                            echo json_encode($result);
                            exit();
                    }
                } elseif ($retries_left == 0) {
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    break;
                } else {
                    $this->log('Retrying to connect to client:'.$_POST['serverHostName'], 'info');
                    $client = $this->client_socket($context, $_POST['serverHostName']);
                    $client->send($sequence);
                }
            }
        }
    }

    public function terminateServerJob(){
        if(empty($_POST['serverHostName'])) {
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
            exit;
        }
        $data = array(
            'action' => 'STOPBACKUP',
            'containerName' => $_POST['containerName']
        );

        $context = new ZMQContext();
        $client = $this->client_socket($context, $_POST['serverHostName']);

        $sequence = 0;
        $retries_left = 3;
        $read = $write = array();

        while ($retries_left) {
            $client->send(json_encode($data));
            $expect_reply = true;
            while ($expect_reply) {

                $poll = new ZMQPoll();
                $poll->add($client, ZMQ::POLL_IN);
                $events = $poll->poll($read, $write, -1);
                --$retries_left;
                if ($events > 0) {

                    $result = $client->recv();

                    switch ($result) {
                        case (200):

                            $response = array(
                                'result' => 'success',
                                'message' => $this->language->get('terminateJob')
                            );
                            echo json_encode($response);
                            exit;

                        case (421):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('421')
                            );
                            echo json_encode($result);
                            exit();
                        case (500):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('500')
                            );
                            echo json_encode($result);
                            exit();
                        case (0):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('0')
                            );
                            echo json_encode($result);
                            exit();
                        default:
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('default')
                            );
                            echo json_encode($result);
                            exit();
                    }
                } elseif ($retries_left == 0) {
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    break;
                } else {
                    $this->log('Retrying to connect to client:'.$_POST['serverHostName'], 'info');
                    $client = $this->client_socket($context, $_POST['serverHostName']);
                    $client->send($sequence);
                }
            }
        }
    }

    public function getSupportLogs(){
        if(empty($_POST['serverHostName'])) {
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
        $data = array(
            'action' => 'SUPPORTLOGS'
        );

        $context = new ZMQContext();
        $client = $this->client_socket($context, $_POST['serverHostName']);

        $sequence = 0;
        $retries_left = 3;
        $read = $write = array();

        while ($retries_left) {
            $client->send(json_encode($data));
            $expect_reply = true;
            while ($expect_reply) {

                $poll = new ZMQPoll();
                $poll->add($client, ZMQ::POLL_IN);
                $events = $poll->poll($read, $write, -1);
                --$retries_left;
                if ($events > 0) {

                    $result = $client->recv();

                    switch ($result) {
                        case (200):

                            $response = array(
                                'result' => 'success',
                                'message' => $this->language->get('generateSupportLogs')
                            );
                            echo json_encode($response);
                            exit;

                        case (421):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('421')
                            );
                            echo json_encode($result);
                            exit();
                        case (500):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('500')
                            );
                            echo json_encode($result);
                            exit();
                        case (0):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('0')
                            );
                            echo json_encode($result);
                            exit();
                        default:
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('default')
                            );
                            echo json_encode($result);
                            exit();
                    }
                } elseif ($retries_left == 0) {
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    break;
                } else {
                    $this->log('Retrying to connect to client:'.$_POST['serverHostName'], 'info');
                    $client = $this->client_socket($context, $_POST['serverHostName']);
                    $client->send($sequence);
                }
            }
        }
    }

    public function restore($deviceUUID, $deviceName, $containerName){

        $headers = array(
            'Content-Type' => 'application/json',
            'token'     => $this->session->get('token'),
            'deviceUUID' => $deviceUUID
        );
        $result = $this->_model->getDeviceDetails($headers);
        if($result['status'] == 200){
            $this->session->put('currentDeviceId', $result['body']->{'BSDeviceInfoElement'}->{'deviceId'});
            $this->session->put('currentDeviceUUID', $result['body']->{'BSDeviceInfoElement'}->{'deviceUUID'});
            $this->session->put('currentDeviceName', $deviceName);
            $this->session->put('containerName', $containerName);
            $data = array(
                'title' => $this->language->get('Server'),
                'crumb1' => $this->language->get('Servers'),
                'active' => 'servers',
                'deviceUUID' => $deviceUUID,
                'deviceName' => $deviceName,
                'containerName' => $containerName,
                'csrf_token' => $this->setCSRFToken(),
                'loggedInUser' => $this->session->all()
            );

            View::renderTemplate('header', $data);
            View::renderTemplate('navigation', $data);
            View::render('server/restore', $data);
            View::renderTemplate('footer', $data);
        }else{
            Url::redirect('404');
        }
    }

    public function getFiles(){
        if(empty($_POST) || empty($_POST['filePath'])) {
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
            exit;
        }
        $filePath = '';
        if($_POST['refresh'] != 'true'){
            $filePath = ($_POST['filePath'] == 'root') ? '' : $_POST['filePath'];
        }
        $headers = array(
            'filePath'      => html_entity_decode($filePath, ENT_QUOTES | ENT_HTML5),
            'deviceId'      => $this->session->get('currentDeviceId'),
            'token'         => $this->session->get('token')
        );

        if(empty($filePath)) {
            $headers['containerName'] = !empty($_POST['containerName']) ? $_POST['containerName'] : '';
        }
        $this->log('Server Files Headers: '.json_encode($headers), 'info');
        $result = $this->_model->getFiles($headers);
        $this->log('Server Files Result: '.json_encode($result), 'info');
        switch ($result['status']) {
            case (200):
                $items = array();
                if(!empty($result['body']->{'backup-file'})){
                    foreach($result['body']->{'backup-file'} as $value){
                        if($value->{'folder'} == 'true'){
                            $folder = array(
                                'backupId' => (string)$value->{'backupId'},
                                'checksum' => (string)$value->{'checksum'},
                                'exists' => (string)$value->{'exists'},
                                'name' => (string)$value->{'file-name'},
                                'user' => (string)$value->{'file-user'},
                                'type' => 'Folder',
                                'icon' => 'folder',
                                'size' => '',
                                'filePath' => (string)$value->{'file-path'},
                                'fullPath' => (!empty((string)$value->{'file-path'})) ? (string)$value->{'file-path'}.'/'.(string)$value->{'file-name'} : (string)$value->{'file-name'},
                                'fsPath' => (string)$value->{'fs-path'},
                                'baseFolder' => (string)$value->{'is-basefolder'},
                                'externalStorage' => (string)$value->{'is-externalStorage'},
                                'clientModifiedTime' => (!empty($value->{'client-modified-time'}) && $value->{'client-modified-time'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'client-modified-time'}) / 1000)) : '',
                                'serverModifiedTime' => (!empty($value->{'server-modified-time'}) && $value->{'server-modified-time'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'server-modified-time'}) / 1000)) : '',
                                'status' => (string)$value->{'status'}
                            );
                            $items = array_merge(array($folder), $items);
                        }else{
                            $fileExt = explode('.', (string)$value->{'file-name'});
                            $file = array(
                                'backupId' => (string)$value->{'backupId'},
                                'checksum' => (string)$value->{'checksum'},
                                'exists' => (string)$value->{'exists'},
                                'name' => (string)$value->{'file-name'},
                                'user' => (string)$value->{'file-user'},
                                'type' => 'File',
                                'filePath' => (string)$value->{'file-path'},
                                'fullPath' => (!empty((string)$value->{'file-path'})) ? (string)$value->{'file-path'}.'/'.(string)$value->{'file-name'} : (string)$value->{'file-name'},
                                'icon' => $this->getFileIcon(end($fileExt)),
                                'fsPath' => (string)$value->{'fs-path'},
                                'baseFolder' => (string)$value->{'is-basefolder'},
                                'storagePlace' => (!empty((string)$value->{'storagePlace'})) ? (string)$value->{'storagePlace'} : 'PG',
                                'externalStorage' => (string)$value->{'is-externalStorage'},
                                'clientModifiedTime' => (!empty($value->{'client-modified-time'}) && $value->{'client-modified-time'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'client-modified-time'}) / 1000)) : '',
                                'serverModifiedTime' => (!empty($value->{'server-modified-time'}) && $value->{'server-modified-time'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'server-modified-time'}) / 1000)) : '',
                                'rawSize' => (int)(string)$value->{'size'},
                                'size' => $this->getFileSize((string)$value->{'size'}),
                                'status' => (string)$value->{'status'}
                            );
                            $items = array_merge(array($file), $items);
                        }
                    }
                }
                if(!empty($items)){
                    $this->session->put('filePath', $filePath);
                }
                $response = array(
                    'result' => 'success',
                    'post' => html_entity_decode($filePath, ENT_QUOTES | ENT_HTML5),
                    'items' => $items
                );
                if($_POST['refresh'] == 'true' && !empty($filePath)){
                    $response['crumb'] = explode('/', $filePath);
                    $response['crumbString'] = $filePath;
                }
                echo json_encode($response);
                exit;
            case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
            case (0):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }

    public function initiateRestore(){
        if(empty($_POST['path'])) {
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => 'Path to restore should not be empty'
                )
            );
            exit;
        }
        if (strpos($_POST['path'], '@') !== false) {
            $newServer = strstr($_POST['path'], '@', true);
            $path = str_replace('@', '', strstr($_POST['path'], '@'));
        }else{
            $newServer = '';
            $path = $_POST['path'];
        }

        $options = array(
            'timestamp' => $_POST['timestamp']
        );
        $data = array(
            'action'            => 'RESTORE',
            'deviceUUID'        => $this->session->get('currentDeviceUUID'),
            'containerName'     => $_POST['containerName'],
            'paths'             => $_POST['paths'],
            'options'           => $options,
            'destinationPath'   => $path
        );

        $this->log('Data'. json_encode($data), 'info');

        $hostName = !empty($newServer) ? $newServer : $this->session->get('currentDeviceName');
        $context = new ZMQContext();
        $client = $this->client_socket($context, $hostName);
        $sequence = 0;
        $retries_left = 3;
        $read = $write = array();
        while ($retries_left) {
            $client->send(json_encode($data));
            $expect_reply = true;
            while ($expect_reply) {
                $poll = new ZMQPoll();
                $poll->add($client, ZMQ::POLL_IN);
                $events = $poll->poll($read, $write, -1);
                --$retries_left;
                if ($events > 0) {

                    $result = $client->recv();
                    switch ($result) {
                        case (200):

                            $response = array(
                                'result' => 'success',
                                'post' => $_POST,
                                'message' => $this->language->get('restoreInitiated')
                            );
                            echo json_encode($response);
                            exit;

                        case (421):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('421')
                            );
                            echo json_encode($result);
                            exit();
                        case (500):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('500')
                            );
                            echo json_encode($result);
                            exit();
                        case (0):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('0')
                            );
                            echo json_encode($result);
                            exit();
                        default:
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('default')
                            );
                            echo json_encode($result);
                            exit();
                    }
                } elseif ($retries_left == 0) {
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    break;
                } else {
                    $this->log('Retrying to connect to client:'.$hostName, 'info');
                    $client = $this->client_socket($context, $hostName);
                    $client->send($sequence);
                }
            }
        }
    }

    public function client_socket(ZMQContext $context, $hostName){
        $client = new ZMQSocket($context, ZMQ::SOCKET_REQ);
        try{
            $client->connect("tcp://".$hostName.":5555");
            $client->setSockOpt(ZMQ::SOCKOPT_LINGER, 0);
            return $client;
        }catch (\Exception $e){
            $this->log('Exception: '. $e, 'info');
            $result = array(
                'result' => 'fail',
                'message' => 'Could not connect to the Server'
            );
            echo json_encode($result);
            exit();
        }
    }

    public function getFileSize($sizeInKB){
        $bytes = ($sizeInKB*1024);
        if ($bytes >= 1073741824){
            $bytes = number_format($bytes / 1073741824, 2) . ' GB';
        }elseif ($bytes >= 1048576){
            $bytes = number_format($bytes / 1048576, 2) . ' MB';
        }elseif ($bytes >= 1024){
            $bytes = number_format($bytes / 1024, 2) . ' KB';
        }elseif ($bytes > 1){
            $bytes = $bytes . ' bytes';
        }elseif ($bytes == 1){
            $bytes = $bytes . ' byte';
        }else{
            $bytes = '0 bytes';
        }
        return $bytes;
    }

    public function getFileIcon($ext){
        switch ($ext) {
            case ('zip'):
                $icon = 'file-archive-o';
                break;
            case ('mp3' || 'mp4'):
                $icon = 'file-audio-o';
                break;
            case ('png' || 'jpeg' || 'jpg' || 'gif'):
                $icon = 'file-photo-o';
                break;
            case ('pdf'):
                $icon = 'file-pdf-o';
                break;
            case ('xlsx' || 'xls'):
                $icon = 'file-excel-o';
                break;
            case ('doc' || 'docx'):
                $icon = 'file-word-o';
                break;
            default:
                $icon = 'file';
                break;
        }
        return $icon;
    }

    public function getOsIcon($deviceType){
        if((strpos($deviceType, 'Wind') === false )){
            if((strpos($deviceType, 'Lin') === false )){
                return 'apple';
            }else{
                return 'linux';
            }
        }else{
            return 'windows';
        }
    }

}
