<?php

namespace Controllers;

use Core\Language;
use Core\View;
use Core\Controller;
use DOMDocument;
use Helpers\Auth;
use Helpers\Url;
use Models\User;
/*
 * Copyright 2014
 * ParaBlu Systems Private Limited
 * All Rights Reserved
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of ParaBlu Systems Private Limited and its suppliers, if any.
 * The intellectual and technical concepts contained herein are proprietary
 * to ParaBlu Systems Private Limited and its suppliers and may be covered by
 * Indian, US and Foreign Patents, patents in process, and are protected by
 * trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from ParaBlu Systems Private Limited.
 *
 * Login controller
 *
 * @author Prashanth Kumar B - prashanth.kumar@parablu.com
 * @version 2.0
 * @date June 10, 2015
 *
 */

class UserController extends Controller{

    /**
     * Below variable contains the object reference of a model
     */

    protected $_model;

    /**
     * Call the parent construct
     */
    public function __construct(){
        parent::__construct();
        $this->auth->loginCheck();
        $this->language->load('user');
        $this->_model = new User();
    }

    /**
     * Display for my account details
     * @param string $username
     */
    public function myAccount($username = ''){
        if($this->session->get('firstLogin1') == "true" && $this->session->get('guest') == "1" ){
            $this->log("hhhhhhhhhhhh","info");
            $visible = 'visible';
        }

        $username = !empty($username) ? htmlentities($username) : $this->session->get('userName');
        $this->auth->checkLoggedInUser($username);
        $userDetails = $this->_model->getUserDetails(array('token' => $this->session->get('token')), $username);

        if(Auth::checkCloudPermission('serverBackup') && Auth::checkUserPermission('admin') == false){
            Url::redirect('servers');
        }

	if($userDetails['body']->{'response-code'} != 500){
            $data = array(
                'title' => $this->language->get('indexPageName'),
                'subPageName' => $this->language->get('profilePage'),
                'active' => 'users',
                'csrf_token' => $this->setCSRFToken(),
                'userDetails' => $userDetails['body'],
                'loggedInUser' => $this->session->all(),
                'popup' => $visible
            );
            
            if($this->session->get('userName') == $userDetails['body']->{'user-name'}){
			$data['lastLoggedInTime'] = $this->session->get('lastLoggedInTime');
		    }
		    else{
			$data['lastLoggedInTime'] = $userDetails['body']->{'lastLoggedInTime'};
		    }

            View::renderTemplate('header', $data);
            View::renderTemplate('navigation', $data);
            View::render('users/profile', $data);
            View::renderTemplate('footer', $data);
        }else{
            Url::redirect('404');
        }

    }

    public function getAllUserDevices(){
        if(!empty($_POST['userName'])){
            $this->valid->filter($_POST, array('userName' => 'trim|sanitize_string'));
            $result = $this->_model->getUserDevices(array('token' => $this->session->get('token'), 'userName' => $_POST['userName']));
            $this->log("RESult".json_encode($result), "info");
            switch ($result['status']) {
                case (200):
                    $devices = [];
                    if($result['body']->bluSyncPortalBackupElement->devices){
                        foreach($result['body']->bluSyncPortalBackupElement->devices as $value){
                            $device = array(
                                'deviceName' => $value->deviceName,
                                'deviceId' => $value->deviceId,
                                'deviceUUID' => $value->deviceUUID,
                                'storageUsedByDevice' => $value->storageUsedByDevice,
                                'windowsDevice' => $value->windowsDevice,
                                'deviceType' => $value->deviceType,
                                'icon' => $this->getOsIcon($value->deviceType),
                                'background' => ($value->windowsDevice == 'true') ? 'purple' : 'primary'
                            );
                            $devices = array_merge(array($device), $devices);
                        }
                    }

                    $used = '';
                    $free = '';
                    $storage = $result['body']->bluSyncPortalBackupElement->storageUsed;
                    $limit = 25;
                    if(stripos($storage, 'MB') != false){
                        $storage = str_replace('MB', '', $storage);
                        $usedInMB = (double)$storage;
                        $used = round(($usedInMB/1024), 2);
                        while($limit < $used){
                            $limit = $limit + 25;
                        }
                        $free = $limit - $used;
                    }elseif(stripos($storage, 'GB') != false){
                        $storage = str_replace('GB', '', $storage);
                        $used = (double)$storage;
                        while($limit < $used){
                            $limit = $limit + 25;
                        }
                        $free = $limit - $used;
                    }
                    if((string)$result['header']->{'container'}['policyconfiguration'] != ""){
                    $response = array(
                        'result' => 'success',
                        'storage' => $storage,
                        'free' => $free,
                        'limit' => $limit,
                        'used' => $used,
                        'devices'  => $devices,
                        'policyconfiguration' => (string)$result['header']->{'container'}['policyconfiguration']
                    );
                    }
                    else {
                    $response = array(
                        'result' => 'success',
                        'storage' => $storage,
                        'free' => $free,
                        'limit' => $limit,
                        'used' => $used,
                        'devices'  => $devices,
                    );    
                    }
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    /**
     * Displays Settings page
     * @param string $username
     */
    public function settings($username = ''){
        $username = !empty($username && $this->session->get('admin') == true) ? htmlentities($username) : $this->session->get('userName');
        $this->auth->checkLoggedInUser($username);
        $userDetails = $this->_model->getUserDetails(array('token' => $this->session->get('token')), $username);
        $backupResult = $this->_model->getAllPolicies(array('token'=>$this->session->get('token')));
        $syncResult = $this->_model->getAllSyncPolicies(array('token'=>$this->session->get('token')));
        $miniClouds = $this->_model->getAllMiniClouds(array('token' => $this->session->get('token')));
        $result = $this->_model->getOdbDetails($headers);

        if(empty((array)$backupResult['body']->{'backupPolicyElementList'}) && $this->auth->checkCloudPermission('backup')){
            header('Location: '.Url::basePath('pms/policy/create'));
            exit;
        }

        if(empty((array)$syncResult['body']->{'syncPolicyElementList'}) && $this->auth->checkCloudPermission('sync')){
            header('Location: '.Url::basePath('pms/sync/policy/create'));
            exit;
        }

        $backupPolicies = [];
        foreach((array)$backupResult['body']->{'backupPolicyElementList'} as $value){
            $policy = array('name' => $value->{'policygroupName'});
            $backupPolicies = array_merge(array($policy), $backupPolicies);
        }

        $syncPolicies = [];
        foreach((array)$syncResult['body']->{'syncPolicyElementList'} as $value){
            $policy = array('name' => $value->{'policygroupName'});
            $syncPolicies = array_merge(array($policy), $syncPolicies);
        }
        if($userDetails['body']->{'response-code'} != 500){

            $data = array(
                'title' => $this->language->get('indexPageName'),
                'subPageName' => $this->language->get('profilePage'),
                'active' => 'users',
                'csrf_token' => $this->setCSRFToken(),
                'activateEdit' => !empty($username && $this->session->get('admin') == true) ? true : false,
                'userDetails' => $userDetails['body'],
                'miniClouds' => $miniClouds['body'],
                'backupPolicies' => $backupPolicies,
                'syncPolicies' => $syncPolicies,
                'loggedInUser' => $this->session->all(),
                'clientID'=> $result['body']->{'clientId'},
                'clientRedirectURI'=> $result['body']->{'clientRedirectURI'},
                'portalURL'=> $result['body']->{'portalURL'},
                'clientSecretKey'=> $result['body']->{'clientSecretKey'},
                'clientResourceURL'=> $result['body']->{'clientResourceURL'},
                'configId'=> $result['body']->{'configId'},
            );
            
            if($this->session->get('userName') == $userDetails['body']->{'user-name'}){
			$data['lastLoggedInTime'] = $this->session->get('lastLoggedInTime');
		    }
		    else{
			$data['lastLoggedInTime'] = $userDetails['body']->{'lastLoggedInTime'};
		    }
		    
            View::renderTemplate('header', $data);
            View::renderTemplate('navigation', $data);
            View::render('users/settings', $data);
            View::renderTemplate('footer', $data);
        }else{
            Url::redirect('404');
        }
    }

    /**
     * Save UserDetails
     * @throws \Helpers\Exception
     */
    public function saveUserDetails(){
        $this->verifyAndCleanItUp();
        $this->auth->checkLoggedInUser($_POST['user_name']);
        if(!empty($_POST)){
            $rules = array(
                'user_name'             => 'required',
                'recovery_key'          => 'required',
                'is_first_login'        => 'required',
                'email_address'         => 'required',
                'account_status'        => 'required',
                'account_type'          => 'required',
                'sync_access'           => 'required',
                'backup'                => 'required',
                'share_enabled'         => 'required'
            );
            $filters = array(
                'password_hash'         => 'trim|sanitize_string',
                'user_name'             => 'trim|sanitize_string',
                'recovery_key'          => 'trim|sanitize_string',
                'is_first_login'        => 'trim|sanitize_string',
                'email_address'         => 'trim|sanitize_string',
                'account_status'        => 'trim|sanitize_string',
                'account_type'          => 'trim|sanitize_string',
                'sync_access'           => 'trim|sanitize_string',
                'backup'                => 'trim|sanitize_string',
                'share_enabled'         => 'trim|sanitize_string'
            );
            $filteredData = $this->valid->filter($_POST, $filters);
            $validate = $this->valid->validate($filteredData, $rules);
            //When the user is using the application, we get Enabled or false and when the admin is using the application, we get true or false
            if($_POST['backup'] == 'Enabled' || $_POST['backup'] == 'true'){
                $_POST['backup'] = 'true';
            } else {
                $_POST['backup'] = 'false';
            }
            
            if($_POST['account_status'] == 'Active'){
                $account_status = 'true';
            } else {
                $account_status = 'false';
            }

            $_POST['displayName'] = preg_replace('/\s+/', '', $_POST['displayName']);
            $_POST['account_type'] = strtolower($_POST['account_type']);
            $_POST['account_type'] = str_replace(' ', '_', $_POST['account_type']);
            if($validate === true){
                $xml = new \SimpleXMLElement('<?xml version="1.0" encoding="UTF-8" standalone="yes"?><user/>');
                $xml->{'is-active'}                     = $account_status;
                $xml->{'is-admin'}                      = (!empty($_POST['account_type'])) ? (($_POST['account_type'] == 'admin') ? 'true' : 'false') : 'false';
                $xml->{'is-readonly-admin'}             = (!empty($_POST['account_type'])) ? (($_POST['account_type'] == 'read_only_admin') ? 'true' : 'false') : 'false';
                $xml->{'backup-enabled'}                = (!empty($_POST['backup'])) ? $_POST['backup'] : 'false';
                $xml->{'server-backup-enabled'}         = (!empty($_POST['serverBackup'])) ? $_POST['serverBackup'] : 'false';
                $xml->{'email-id'}                      = (!empty($_POST['email_address'])) ? $_POST['email_address'] : null;
                $xml->{'is-first-login'}                = (!empty($_POST['is_first_login'])) ? $_POST['is_first_login'] : 'false';
                $xml->{'last-modified-timestamp'}       = time();
                $xml->{'migrationtool-enabled'}         = (!empty($_POST['account_type'])) ? (($_POST['account_type'] == 'admin') ? 'true' : 'false') : 'false';
                if(!empty($_POST['password_hash'])){
                $xml->{'password-hash'}                 = $_POST['password_hash'];
                }
                $xml->{'is-guest'}                 = $_POST['is_guest'];
                if(!empty($_POST['local_user'])){
                $xml->{'localUser'}                 = $_POST['local_user'];
                }
                $xml->{'pms-enabled'}                   = (!empty($_POST['account_type'])) ? (($_POST['account_type'] == 'admin') ? 'true' : 'false') : 'false';
                $xml->{'report-enabled'}                = (!empty($_POST['account_type'])) ? (($_POST['account_type'] == 'admin' || $_POST['account_type'] == 'read_only_admin') ? 'true' : 'false') : 'false';
                $xml->{'recovery-key'}                  = (!empty($_POST['recovery_key'])) ? $_POST['recovery_key'] : null;
                $xml->{'share-enabled'}                 = (!empty($_POST['share_enabled'])) ? $_POST['share_enabled'] : 'false';
                $xml->{'support-enabled'}               = (!empty($_POST['account_type'])) ? (($_POST['account_type'] == 'admin' || $_POST['account_type'] == 'read_only_admin') ? 'true' : 'false') : 'false';
                $xml->{'sync-enabled'}                  = (!empty($_POST['sync_access'])) ? $_POST['sync_access'] : 'false';
                $xml->{'user-id'}                       = (!empty($_POST['user_id'])) ? $_POST['user_id'] : null;
                $xml->{'user-name'}                     = (!empty($_POST['user_name'])) ? $_POST['user_name'] : null;
                if(!empty($_POST['displayName'])){
                $xml->{'display-name'}                  = $_POST['displayName'];
                }
                if(!empty($_POST['location'])){
                $xml->{'location'}                      = $_POST['location'];
                }
                if(!empty($_POST['department'])){
                $xml->{'department'}                    = $_POST['department'];
                }
                if($_POST['backup'] == 'true'){
                $xml->{'policy-name'}                   = (!empty($_POST['policyName'])) ? $_POST['policyName'] : null;
                }
                $xml->{'sync-policy-name'}              = (!empty($_POST['syncPolicyName'])) ? $_POST['syncPolicyName'] : null;
                $xml->{'odb-loginId'}                   = (!empty($_POST['odb_login_Id'])) ? $_POST['odb_login_Id'] : '';
		$xml->{'last-Logged-In-Time'}                 = $_POST['lastLoginTime'];
                $xml->addChild('mini_clouds');
                if(!empty($_POST['miniCloudName'])){
                    $mini_clouds = (is_array($_POST['miniCloudName'])) ? $_POST['miniCloudName'] : explode(',', $_POST['miniCloudName']);
                    foreach($mini_clouds as $value){
                        $mini = $xml->mini_clouds->addChild('mini-cloud');
                        $mini->addChild('mini-cloud-name', htmlspecialchars($value));
                    }
                }
                $userString = $xml->asXML();
                $userString = str_replace('mini_clouds', 'mini-clouds', $userString);
                $this->log(json_encode($_POST), "info");
                $this->log($userString, "info");

                $headers = array(
                    'Content-type' => 'application/xml',
                    'token' => $this->session->get('token')
                );
                $result = $this->_model->updateUserDetails($headers, $userString, $this->session->get('userName'));

                switch ($result['status']) {
                    case (200):
                        $response = array(
                            'result' => 'success',
                            'message' => $this->language->get('userSavedSuccess'),
                            'data' => (string)$result['body']->{'is-active'}
                        );
                        echo json_encode($response);
                        exit;
                    case (201):
                        $response = array(
                            'result' => 'success',
                            'message' => $this->language->get('userSavedSuccess')
                        );
                        echo json_encode($response);
                        exit;

                    case (412):

                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('412')
                        );
                        echo json_encode($result);
                        exit();

                    case (447):
                        $response = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('447')
                        );
                        echo json_encode($response);
                        exit;

                    case (442):
                        $response = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('442')
                        );
                        echo json_encode($response);
                        exit;
                    case (421):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('421')
                        );
                        echo json_encode($result);
                        exit();
                    case (403):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('403')
                        );
                        echo json_encode($result);
                        exit();
                    case (500):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('500')
                        );
                        echo json_encode($result);
                        exit();
                    case (0):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('0')
                        );
                        echo json_encode($result);
                        exit();
                    default:
                        $result = array(
                            'result' => 'fail',
                            'code' => $result['status'],
                            'msg' => $this->language->get('default')
                        );
                        echo json_encode($result);
                        exit();
                }
            }else{
                $result = array(
                    'result' => 'fieldFail',
                    'msg' => $this->valid->get_errors_array()
                );
                echo json_encode($result);
            }
        }else{
            $result = array(
                'result' => 'fail',
                'msg' => $this->language->get('noData')
            );
            echo json_encode($result);
        }
    }

    /**
     * Returns all the miniClouds
     */
    public function getAllMiniClouds(){
        $headers = array(
            'token' => $this->session->get('token')
        );
        $result = $this->_model->getAllMiniClouds($headers);
        if($result['status'] == 200){
            print_r($result['body']);
        }else{
            echo json_encode(array('result' => 'fail', 'msg' => 'Could not find any Mini Clouds'));
        }
    }

    /**
     * Uploads the profile picture
     */
    public function upload(){
        if (!empty($_FILES)) {
            list($type) = explode('/', mime_content_type($_FILES['file']['tmp_name']));
            if($type == 'image'){
                $targetPath = 'app/templates/default/uploads/profile';
                $nameMD5 = md5($this->session->get('userName'));
                $targetFile =  $targetPath.DS.$nameMD5;
                $ext = (explode(".", $_FILES['file']['name']));
                if(file_exists($targetFile.'.'.end($ext))) {
                    unlink($targetFile.'.'.end($ext));
                }
                if(move_uploaded_file($_FILES['file']['tmp_name'], $targetFile.'.'.end($ext))){
                    header('HTTP/1.1 200 uploaded');
                    header('Content-Type: application/json; charset=UTF-8');
                    die(json_encode(array('message' => 'success', 'code' => 200)));
                }else{
                    header('HTTP/1.1 500 Technical Issue');
                    header('Content-Type: application/json; charset=UTF-8');
                    die(json_encode(array('message' => 'ERROR in uploading, only ".jpg" files are allowed', 'code' => 500)));
                }
            }else{
                header('HTTP/1.1 500 Technical Issue');
                header('Content-Type: application/json; charset=UTF-8');
                die(json_encode(array('message' => 'ERROR in uploading, Invalid file type', 'code' => 500)));
            }
        }else{
            header('HTTP/1.1 404 File not found');
            header('Content-Type: application/json; charset=UTF-8');
            die(json_encode(array('message' => 'ERROR in uploading, only ".jpg" files are allowed', 'code' => 500)));
        }
    }

    /**
     * Below method accepts username param
     * and blocks the user
     * @throws \Helpers\Exception
     */
    public function block(){
        Auth::allowOnly('admin');
        if(empty($_POST['userName']) && empty($_POST['blockUser'])){
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
            exit;
        }
        if($_POST['userName'] != $this->session->get('userName')){
            $this->valid->filter($_POST, array('userName' => 'trim|sanitize_string'));
            $headers = array(
                'token' => $this->session->get('token'),
                'blocked' => $_POST['blockUser'],
                'userName' => $this->session->get('userName')
            );
            $result = $this->_model->blockUser($headers, $_POST['userName']);

            switch ($result['status']) {
                case (200):
                    if($_POST['blockUser'] == 'true'){
                        $msg = $this->language->get('userUnBlocked');
                    }else {
                        $msg = $this->language->get('userBlocked');
                    }
                    if($_POST['blockUser'] == 'false'){
                        $emailProps = array(
                            'actionOn' => $_POST['userName'],
                            'action' => 'user-blocked',
                            'toAdmins' => true,
                            'actionBy' => $this->session->get('userName'),
                            'cloudName' => $this->session->get('cloudName'),
                        );
                        if(EMAIL_ONLY_ADMIN == false){
                            $userDetails = $this->_model->getUserDetails(array('token' => $this->session->get('token')), $_POST['userName']);
                            $emailProps['recipents'] = array(
                                array(
                                    'userName' => $_POST['userName'],
                                    'email' => !empty((string)$userDetails['body']->{'email-id'})? (string)$userDetails['body']->{'email-id'} : 'support@parablu.com'
                                )
                            );
                        }else{
                            $emailProps['recipents'] = array(
                                array(
                                    'userName' => $_POST['userName'],
                                    'email' => 'support@parablu.com'
                                )
                            );
                        }
                        $this->sendMail($emailProps);
                    }
                    $response = array(
                        'result' => 'success',
                        'message' => $msg,
                        'data' => (string)$result['body']->{'is-active'}
                    );
                    echo json_encode($response);
                    exit;
                case (201):
                    $response = array(
                        'result' => 'success',
                        'message' => $this->language->get('userBlocked')
                    );
                    echo json_encode($response);
                    exit;
                case (447):
                    $response = array(
                        'result' => 'fail',
                        'message' => $this->language->get('447')
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (417):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('417')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'cannot','message' => $this->language->get('cannotBlock')));
        }
    }


    /**
     * Below method marks a user for delete
     * @throws \Helpers\Exception
     */
    public function delete(){
        Auth::allowOnly('admin');
        if(!empty($_POST['userName']) && !empty($_POST['blockUser'])){
            if($_POST['userName'] != $this->session->get('userName')){
                $this->valid->filter($_POST, array('userName' => 'trim|sanitize_string'));

                $headers = array(
                    'token' => $this->session->get('token'),
                    'userName' => $this->session->get('userName')
                );
                $result = $this->_model->deleteUser($headers, $_POST['userName']);

                switch ($result['status']) {
                    case (200):

                        $emailProps = array(
                            'actionOn' => $_POST['userName'],
                            'action' => 'user-delete',
                            'toAdmins' => true,
                            'actionBy' => $this->session->get('userName'),
                            'cloudName' => $this->session->get('cloudName'),
                        );
                        if(EMAIL_ONLY_ADMIN == false){
                            $userDetails = $this->_model->getUserDetails(array('token' => $this->session->get('token')), $_POST['userName']);
                            $emailProps['recipents'] = array(
                                array(
                                    'userName' => $_POST['userName'],
                                    'email' => !empty((string)$userDetails['body']->{'email-id'})? (string)$userDetails['body']->{'email-id'} : 'support@parablu.com'
                                )
                            );
                        }else{
                            $emailProps['recipents'] = array(
                                array(
                                    'userName' => $_POST['userName'],
                                    'email' => 'support@parablu.com'
                                )
                            );
                        }
                        $this->sendMail($emailProps);

                        $response = array(
                            'result' => 'success',
                            'message' => $this->language->get('userDeleted'),
                            'data' => (string)$result['body']->{'is-active'}
                        );
                        echo json_encode($response);
                        exit;
                    case (201):
                        $response = array(
                            'result' => 'success',
                            'message' => $this->language->get('userDeleted')
                        );
                        echo json_encode($response);
                        exit;
                    case (447):
                        $response = array(
                            'result' => 'fail',
                            'message' => $this->language->get('447')
                        );
                        echo json_encode($response);
                        exit;
                    case (448):
                        $response = array(
                            'result' => 'fail',
                            'message' => $this->language->get('448')
                        );
                        echo json_encode($response);
                        exit;
                    case (421):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('421')
                        );
                        echo json_encode($result);
                        exit();
                    case (417):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('417')
                        );
                        echo json_encode($result);
                        exit();
                    case (500):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('500')
                        );
                        echo json_encode($result);
                        exit();
                    case (0):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('0')
                        );
                        echo json_encode($result);
                        exit();
                    default:
                        $result = array(
                            'result' => 'fail',
                            'code' => $result,
                            'msg' => $this->language->get('default')
                        );
                        echo json_encode($result);
                        exit();
                }
            }else{
                echo json_encode(array('result' => 'cannot','message' => $this->language->get('cannotDelete')));
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    /**
     * Renders users page
     */
    public function usersBulkBlockDelete(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'subTitle' => $this->language->get('usersBulkBlockDelete'),
            'crumb1' => $this->language->get('usersBulkBlockDelete'),
            'active' => 'users',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('users/usersBulkBlockDelete', $data);
        View::renderTemplate('footer', $data);
    }

    /**
     * Renders users page
     */
    public function users(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'subTitle' => $this->language->get('listOfUsers'),
            'active' => 'users',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('users/users', $data);
        View::renderTemplate('footer', $data);
    }

    /**
     * Below method retrieves all the users
     */
    public function getAllUsers(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){

            $headers = array(
                'token'     => $this->session->get('token'),
                'skipValue' => !empty($_POST['skipValue']) ? $_POST['skipValue'] : 0
            );

            if(!empty($_POST['userName']) && $_POST['userName'] != 'all'){
                $headers['userName'] = $_POST['userName'];
                $result = $this->_model->searchForAUser($headers);
            }else{
                $result = $this->_model->getAllUsers($headers);
            }
            $this->log('Users Result'. json_encode($result), 'info');
            switch ($result['status']) {
                case (200):
                    $users = array();
                    foreach($result['body'] as $value){
                        $user = array(
                            'profilePic'       => Url::profilePic((string)$value->{'user-name'}),
                            'userGroup'        => !empty((string)$value->{'user-group'}) ? (string)$value->{'user-group'} : '-',
                            'emailId'          => (string)$value->{'email-id'},
                            'isActive'         => ((string)$value->{'is-active'} == 'true') ? true : false,
                            'userName'         => (string)$value->{'user-name'},
                            'displayName'         => (string)$value->{'display-name'},
                            'isAdmin'          => ((string)$value->{'is-admin'} == 'true') ? true : false,
                            'time'        => (string)$value->{'last-modified-timestamp'},
                            'isReadonlyAdmin' => ((string)$value->{'is-readonly-admin'} == 'true') ? true : false
                        );
                        $users = array_merge(array($user), $users);
                    }
                    $response = array(
                        'result' => 'success',
                        'users'  => array_reverse($users)
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    /**
     * Below method retrieves all the users
     */
    public function getAllUserNames(){
        if(empty($_POST)){
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
            exit;
        }
        $headers = array(
            'Accept' => 'application/json',
            'token'     => $this->session->get('token')
        );
        $result = $this->_model->getAllNames($headers);
        switch ($result['status']) {
            case (200):
                $users = [];
                if(!empty($result['body']->{'stringElementList'})){
                    foreach($result['body']->{'stringElementList'} as $key => $value){
                        if($key < 10){
                            $userName = array('id' => $key, 'text' => $value->{'name'});
                            $users = array_merge(array($userName), $users);
                        }
                    }
                }

                $response = array(
                    'result' => 'success',
                    'users'  => array_reverse($users)
                );
                echo json_encode($response);
                exit;
            case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
            case (0):
                $result = array(
                    'result' => 'fail',
                    'users'  => $result['body'],
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }

    public function deviceDetails(){
        if(!empty($_POST) && !empty($_POST['deviceUUID'])){

            $headers = array(
                'Content-Type' => 'application/json',
                'token'     => $this->session->get('token'),
                'deviceUUID' => $_POST['deviceUUID']
            );

            $result = $this->_model->getDeviceDetails($headers);

            switch ($result['status']) {
                case (200):

/*                    $result['body']->{'BSDeviceInfoElement'}->{'icon'} = $this->getOsIcon($result['body']->{'BSDeviceInfoElement'}->{'osType'});
                    $result['body']->{'BSDeviceInfoElement'}->{'os'} = $this->getOsType($result['body']->{'BSDeviceInfoElement'}->{'osType'});
*/
if($result['body']->{'BSDeviceInfoElement'}->{'windowsDevice'} == true && preg_match('/bit/',$result['body']->{'BSDeviceInfoElement'}->{'osVersion'})) {
$version = $result['body']->{'BSDeviceInfoElement'}->{'osVersion'};
$char = "(";
$version = $result['body']->{'BSDeviceInfoElement'}->{'osVersion'};
                    $str = explode($char, $version);
                    $version = "(".$str[1];
$result['body']->{'BSDeviceInfoElement'}->{'version'} = $version;
}

                    $result['body']->{'BSDeviceInfoElement'}->{'icon'} = $this->getOsIcon($result['body']->{'BSDeviceInfoElement'}->{'osType'});
                    $result['body']->{'BSDeviceInfoElement'}->{'os'} = $this->getOsType($result['body']->{'BSDeviceInfoElement'}->{'osType'});

                    $response = array(
                        'result' => 'success',
                        'deviceDetails'  => $result['body']->{'BSDeviceInfoElement'}
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    public function changePassword(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST['password'])){

            $result = $this->_model->getUserData(array('token' => $this->session->get('token')), $this->session->get('userName'));
            $xml1   = new \SimpleXMLElement($result['body']);
            $oldpass = $xml1->{'password-hash'};
                    $this->log('passwordOld: '. $xml1->{'password-hash'}, 'info');
                   // $this->log('password123: '. json_encode($result), 'info');
            switch ($result['status']) {
                case (200):

                    $xml    = new \SimpleXMLElement($result['body']);
                    $xml->{'password-hash'} = hash('sha512', $_POST['password']);
                    $newpass = $xml->{'password-hash'};
                    $this->log('passwordNew: '. $xml->{'password-hash'}, 'info');
                    $this->log('gggggg: '. strcmp($newpass,$oldpass), 'info');
                    $oldpassword_valuepost = hash('sha512', $_POST['oldpassword']);
                    $similarity1 = strcmp($oldpassword_valuepost,$oldpass);
                    $this->log('gggggg1: '. strcmp($oldpassword_valuepost,$oldpass), 'info');
                    $similarity = strcmp($newpass,$oldpass);
                    if($similarity1 != 0){
                        $response = array(
                            'result' => 'oldNotMatched'
                        );
                        echo json_encode($response);
                        exit;
                    }
                    if($similarity == 0 ){
                        
                        $response = array(
                            'result' => 'fail',
                            'msg'  => 'passwordnotchanged'
                        );
                        echo json_encode($response);
                        exit;
                    }
                    
                    else {
                    $userDetails    = $xml->asXML();
                    $userUpdateResult = $this->_model->updateUserDetails(
                        array('Content-type' => 'application/xml','token' => $this->session->get('token')),
                        $userDetails, $this->session->get('userName')
                    );
                    if($userUpdateResult['status'] == 200){
                        $response = array(
                            'result' => 'success',
                            'msg'  => $this->language->get('passwordChanged')
                        );
                        echo json_encode($response);
                        exit;
                    }else{
                        $response = array(
                            'result' => 'fail',
                            'msg'  => $this->language->get('default')
                        );
                        echo json_encode($response);
                        exit;
                    }
            }
            
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }
    
    public function changePasswordForGuest(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST['passwordGuest'])){
            $result = $this->_model->getUserData(array('token' => $this->session->get('token')), $this->session->get('userName'));
            $xml1   = new \SimpleXMLElement($result['body']);
            $oldpass = $xml1->{'password-hash'};
                    $this->log('passwordOld: '. $xml1->{'password-hash'}, 'info');
                   // $this->log('password123: '. json_encode($result), 'info');
            switch ($result['status']) {
                case (200):

                    $xml    = new \SimpleXMLElement($result['body']);
                    $xml->{'password-hash'} = hash('sha512', $_POST['passwordGuest']);
                    $newpass = $xml->{'password-hash'};
                    $this->log('passwordNew: '. $xml->{'password-hash'}, 'info');
                    $this->log('gggggg: '. strcmp($newpass,$oldpass), 'info');
                    $similarity = strcmp($newpass,$oldpass);
                    if($similarity == 0){
                        
                        $response = array(
                            'result' => 'fail',
                            'msg'  => 'passwordnotchanged'
                        );
                        echo json_encode($response);
                        exit;
                    }
                    else {
                    $userDetails    = $xml->asXML();
                    $userUpdateResult = $this->_model->updateUserDetails(
                        array('Content-type' => 'application/xml','token' => $this->session->get('token')),
                        $userDetails, $this->session->get('userName')
                    );
                    if($userUpdateResult['status'] == 200){
                        $response = array(
                            'result' => 'success',
                            'msg'  => $this->language->get('passwordChanged')
                        );
                        echo json_encode($response);
                        exit;
                    }else{
                        $response = array(
                            'result' => 'fail',
                            'msg'  => $this->language->get('default')
                        );
                        echo json_encode($response);
                        exit;
                    }
            }
            
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    public function sendMail($emailProps){

        $headers = array(
            'Content-Type' => 'application/json',
            'token' => $this->session->get('token')
        );

        return $this->_model->sendMail($headers, $emailProps);

    }

    /**
     * Renders createUser Page
     */
    public function createUser(){
        Auth::allowOnly('admin');
        $backupResult = $this->_model->getAllPolicies(array('token'=>$this->session->get('token')));
        $syncResult = $this->_model->getAllSyncPolicies(array('token'=>$this->session->get('token')));
        $headers = array(
            'Content-type' => 'application/xml',
            'Accept' => 'application/xml',
            'Accept-Charset' => 'utf-8',
            'token' => $this->session->get('token'),
            'cloudParams' => ''
        );
        $cloudLicense = $this->_model->getCloudLicense($headers);
        $xml = new \SimpleXMLElement($cloudLicense['body']);
        $miniClouds = $this->_model->getAllMiniClouds(array('token' => $this->session->get('token')));

        if(empty((array)$backupResult['body']->{'backupPolicyElementList'}) && $this->auth->checkCloudPermission('backup')){
            header('Location: '.Url::basePath('pms/policy/create'));
            exit;
        }

        if(empty((array)$syncResult['body']->{'syncPolicyElementList'}) && $this->auth->checkCloudPermission('sync')){
            header('Location: '.Url::basePath('pms/sync/policy/create'));
            exit;
        }

        $backupPolicies = [];
        foreach((array)$backupResult['body']->{'backupPolicyElementList'} as $value){
            $policy = array('name' => $value->{'policygroupName'});
            $backupPolicies = array_merge(array($policy), $backupPolicies);
        }

        $syncPolicies = [];
        foreach((array)$syncResult['body']->{'syncPolicyElementList'} as $value){
            $policy = array('name' => $value->{'policygroupName'});
            $syncPolicies = array_merge(array($policy), $syncPolicies);
        }

        $data = array(
            'title' => $this->language->get('indexPageName'),
            'subTitle' => $this->language->get('create'),
            'active' => 'users',
            'license' => $xml->{'license'},
            'csrf_token' => $this->setCSRFToken(),
            'miniClouds' => $miniClouds['body'],
            'backupPolicies' => $backupPolicies,
            'syncPolicies' => $syncPolicies,
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('users/create', $data);
        View::renderTemplate('footer', $data);
    }

    public function saveNewUserDetails(){
        $this->verifyAndCleanItUp();
        Auth::allowOnly('admin');
        if(!empty($_POST['username']) && !empty($_POST['email']) && !empty($_POST['password'])){
            if($_POST['account_type'] == 'read_only_admin'){
                $isAdmin = 'false';
                $isReadAdmin = 'true';
            }elseif($_POST['account_type'] == 'admin'){
                $isAdmin = 'true';
                $isReadAdmin = 'false';
            }else{
                $isAdmin = 'false';
                $isReadAdmin = 'false';
            }

            $_POST['username'] = preg_replace('/\s+/', '', $_POST['username']);
            $_POST['displayName'] = preg_replace('/\s+/', '', $_POST['displayName']);
            $values = array(
                'userName' => htmlspecialchars_decode($_POST['username'],ENT_QUOTES),
                'displayName' => $_POST['displayName'],
                'location' => $_POST['location'],
                'department' => $_POST['department'],
                'password' => hash('sha512', $_POST['password']),
                'emailId' => $_POST['email'],
                'isAdmin' => !empty($isAdmin) ? $isAdmin : 'false',
                'policyName' => (!empty($_POST['policyName'])) ? $_POST['policyName'] : 'false',
                'odb-loginId' => (!empty($_POST['odb_login_Id'])) ? $_POST['odb_login_Id'] : '',
                'syncPolicyName' => (!empty($_POST['syncPolicyName'])) ? $_POST['syncPolicyName'] : 'false',
                'readOnlyAdmin' => !empty($isReadAdmin) ? $isReadAdmin : 'false',
                'isSyncEnabled' => !empty($_POST['sync_access']) ? $_POST['sync_access'] : 'false',
                'isBackupEnabled' => !empty($_POST['backup']) ? $_POST['backup'] : 'false',
                'isServerBackupEnabled' => !empty($_POST['serverBackup']) ? $_POST['serverBackup'] : 'false',
                'isShareEnabled' => !empty($_POST['share_enabled']) ? $_POST['share_enabled'] : 'false'
            );
            $userDetails = $this->getUserXML($_POST['miniCloudName'], $values);
            $this->log('User Result: '.$userDetails, 'info');
            $headers = array(
                'Content-type' => 'application/xml',
                'Accept'       => 'application/xml',
                'Accept-Charset'=> 'utf-8',
                'token'=> $this->session->get('token')
            );

            $result = $this->_model->createNewUser($headers, $this->session->get('userName'), $userDetails);
            $this->log('Result: '. json_encode($result), 'info');

            switch ($result['status']) {
                case (200):

                    $result = array(
                        'result' => 'success',
                        'msg' => 'User has been created Successfully'
                    );
                    echo json_encode($result);
                    exit();

                case (417):

                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('417')
                    );
                    echo json_encode($result);
                    exit();

                case (409):

                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('409')
                    );
                    echo json_encode($result);
                    exit();

                case (412):

                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('412')
                    );
                    echo json_encode($result);
                    exit();

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }


    /**
     * Below method retrieves all the users
     */
    public function searchUsers(){
        if(!empty($_GET)){

            $headers = array(
                'token'     => $this->session->get('token'),
                'skipValue' => !empty($_GET['skipValue']) ? $_GET['skipValue'] : 0
            );

            if(!empty($_GET['q']) && $_GET['q'] != 'all'){
                $headers['userName'] = $_GET['q'];
                $result = $this->_model->searchForAUser($headers);
            }else{
                $result = $this->_model->getAllUsers($headers);
            }

            switch ($result['status']) {
                case (200):
                    $users = array();
                    $users = array_merge(array(array("userName" => "all", "profilePic" => Url::profilePic('default'))), $users);
                    foreach($result['body'] as $value){
                        $user = array(
                            'profilePic'       => Url::profilePic((string)$value->{'user-name'}),
                            'userName'         => (string)$value->{'user-name'}
                        );
                        $users = array_merge(array($user), $users);
                    }
                    $response = array(
                        'result' => 'success',
                        'users'  => array_reverse($users)
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }


    /**
     * Below method retrieves all the users
     */
    public function searchUserWithDevice(){
        if(!empty($_GET)){

            $headers = array(
                'token'     => $this->session->get('token'),
                'skipValue' => !empty($_GET['skipValue']) ? $_GET['skipValue'] : 0
            );

            $users = array();
            if($_GET['q'] == "all"){
                $users = array_merge(array(array("userName" => "all", "profilePic" => Url::profilePic('default'))), $users);
                $response = array(
                    'result' => 'success',
                    'users'  => array_reverse($users)
                );
                echo json_encode($response);
                exit;
            }

            $headers['userName'] = $_GET['q'];
            $result = $this->_model->searchUserWithDevice($headers);
            $this->log($result);
            switch ($result['status']) {
                case (200):
                    $users = array_merge(array(array("userName" => "all", "profilePic" => Url::profilePic('default'))), $users);
                    foreach($result['body']->{"searchUserElementList"} as $value){
                        $user = array(
                            'profilePic'       => Url::profilePic((string)$value->{'userName'}),
                            'userName'         => (string)$value->{'userName'},
                            'devices'          => (array)$value->{'deviceNameList'}
                        );
                        $users = array_merge(array($user), $users);
                    }
                    $response = array(
                        'result' => 'success',
                        'users'  => array_reverse($users)
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    /**
     * Below method retrieves all the UnBlocked users
     */
    public function getAllUnBlockedUsers(){
        if(!empty($_GET)){

            $headers = array(
                'token'     => $this->session->get('token'),
                'skipValue' => !empty($_GET['skipValue']) ? $_GET['skipValue'] : 0
            );

            if(!empty($_GET['q']) && $_GET['q'] != 'all'){
                $headers['userName'] = $_GET['q'];
                $result = $this->_model->searchForAUser($headers);
            }else{
                $result = $this->_model->getAllUsers($headers);
            }

            switch ($result['status']) {
                case (200):
                    $users = array();
                    foreach($result['body'] as $value){
                        if((string)$value->{'is-active'} == 'true' && (string)$value->{'user-name'} != $this->session->get('userName')){
                            $user = array(
                                'profilePic'       => Url::profilePic((string)$value->{'user-name'}),
                                'userName'         => (string)$value->{'user-name'}
                            );
                            $users = array_merge(array($user), $users);
                        }
                    }
                    $response = array(
                        'result' => 'success',
                        'users'  => array_reverse($users)
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    /**
     * Below method retrieves all the UnBlocked users
     */
    public function getAllUnBlockedUsersWithLoggedInUser(){
        if(!empty($_GET)){

            $headers = array(
                'token'     => $this->session->get('token'),
                'skipValue' => !empty($_GET['skipValue']) ? $_GET['skipValue'] : 0
            );

            if(!empty($_GET['q']) && $_GET['q'] != 'all'){
                $headers['userName'] = $_GET['q'];
                $result = $this->_model->searchForAUser($headers);
            }else{
                $result = $this->_model->getAllUsers($headers);
            }

            switch ($result['status']) {
                case (200):
                    $users = array();
                    foreach($result['body'] as $value){
                        if((string)$value->{'is-active'} == 'true'){
                            $user = array(
                                'profilePic'       => Url::profilePic((string)$value->{'user-name'}),
                                'userName'         => (string)$value->{'user-name'}
                            );
                            $users = array_merge(array($user), $users);
                        }
                    }
                    $response = array(
                        'result' => 'success',
                        'users'  => array_reverse($users)
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    /**
     * Below method retrieves all the Blocked users
     */
    public function getAllBlockedUsers(){
        if(!empty($_GET)){

            $headers = array(
                'token'     => $this->session->get('token'),
                'skipValue' => !empty($_GET['skipValue']) ? $_GET['skipValue'] : 0
            );

            if(!empty($_GET['q']) && $_GET['q'] != 'all'){
                $headers['userName'] = $_GET['q'];
                $result = $this->_model->searchForAUser($headers);
            }else{
                $result = $this->_model->getAllUsers($headers);
            }

            switch ($result['status']) {
                case (200):
                    $users = array();
                    foreach($result['body'] as $value){
                        if((string)$value->{'is-active'} == 'false' && (string)$value->{'user-name'} != $this->session->get('userName')){
                            $user = array(
                                'profilePic'       => Url::profilePic((string)$value->{'user-name'}),
                                'userName'         => (string)$value->{'user-name'}
                            );
                            $users = array_merge(array($user), $users);
                        }
                    }
                    $response = array(
                        'result' => 'success',
                        'users'  => array_reverse($users)
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    /**
     * Below method retrieves all the UnBlocked users
     */
    public function getAllUnBlockedUsersWithGroups(){
        if(empty($_GET)){
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
        $headers = array(
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
            'token'     => $this->session->get('token'),
            'skipValue' => !empty($_GET['skipValue']) ? $_POST['skipValue'] : 0
        );
        if(!empty($_GET['q']) && $_GET['q'] != 'all'){
            $headers['userName'] = $_GET['q'];
            $result = $this->_model->searchForAUserWithGroup($headers);
        }else{
            $result = $this->_model->getAllUsersWithGroup($headers);
        }

        switch ($result['status']) {
            case (200):
                $users = array();
                if(!empty($result['body']->{'stringElementList'})){
                    foreach($result['body']->{'stringElementList'} as $value){
                        $user = array(
                            'profilePic'       => Url::profilePic((string)$value->{'name'}),
                            'userName'         => (string)$value->{'name'},
                            'emailId'         => (!empty((string)$value->{'emailId'})) ? (string)$value->{'emailId'} : (string)$value->{'name'}
                        );
                        $users = array_merge(array($user), $users);

                    }
                }

                $response = array(
                    'result' => 'success',
                    'users'  => (!empty($users)) ? array_reverse($users) : $users
                );
                echo json_encode($response);
                exit;
            case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
            case (0):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }

    /**
     * Below method Blocks list of users
     */
    public function blockMultipleUsers(){
        if(!empty($_POST['users'])){
             $this->log('UUUUSSSSEEEERRRRRSSSSS'. $_POST['users'], 'info');
            $headers = array(
                'Content-Type' => 'application/json',
                'token'     => $this->session->get('token'),
                'active'    => $_POST['active'],
                'blockedBy'    => $this->session->get('userName'),
                );
            
            $this->log(json_encode($headers), "info");
            $this->log(json_encode($_POST['users']), "info");
            $result = $this->_model->blockMultipleUsers($headers,$_POST['users'] );
            $this->log('Type'. $_POST['active'], 'info');
            switch ($result['status']) {
                case (200):
                    if($_POST['active'] == 'true'){
                        $msg = $this->language->get('unBlockedMultipleUsers');
                    }else{
                        $msg = $this->language->get('blockedMultipleUsers');
                    }
                    $response = array(
                        'result' => 'success',
                        'msg' => $msg,
                        'users'  => $result
                    );
                    echo json_encode($response);
                    exit;
                case (447):
                    $response = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('447')
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (417):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('417')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }
    
    public function blockUsers(){
        if(!empty($_POST['users'])){
            $headers = array(
                'Content-type' => 'application/json',
                'token'     => $this->session->get('token'),
            );
            
            $this->log('Type________'. json_encode($data), 'info');
            $result = $this->_model->blockUsers($headers, $_POST['users']);
            $this->log('Type'. $result['body']->userElementList, 'info');
            switch ($result['status']) {
                case (200):
                    $this->log('UUUUUUUEEEEE'. json_encode($result['body']), 'info');
                    $policies = [];
                    foreach($result['body']->{'userElementList'} as $value){
                        $policy = array(
                          
                            'userName' => (string)$value->{'userName'},
                            'emailId' => (string)$value->{'emailId'},
                            'isActive' => $value->{'active'},
                            'time'        =>!empty($value->{'lastModifiedTimestamp'})? date("M j, Y, g:i:s", (((int)$value->{'lastModifiedTimestamp'}) / 1000)) : '',
                            'status' => (string)$value->{'status'},
                            
                            );
                        $policies = array_merge(array($policy), $policies);
                    }
                    
                    $this->log('UUUUUUU'. json_encode($policies), 'info');
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('deletedMultipleUsers'),
                        'naveen'  => array_reverse($policies)
                    );
                    echo json_encode($response);
                    exit;
                case (447):
                    $response = array(
                        'result' => 'fail',
                        'message' => $this->language->get('447')
                    );
                    echo json_encode($response);
                    exit;
                
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    /**
     * Below method Delete list of users
     */
    public function deleteMultipleUsers(){
        if(!empty($_POST['users'])){
            $headers = array(
                'Content-Type' => 'application/json',
                'token'     => $this->session->get('token'),
                'deletedBy'    => $this->session->get('userName'),
            );
            $this->log(json_encode($headers), "info");
            $this->log(json_encode($_POST['users']), "info");
            $result = $this->_model->deleteMultipleUsers($headers, $_POST['users']);

            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('deletedMultipleUsers'),
                        'users'  => $result
                    );
                    echo json_encode($response);
                    exit;
                case (447):
                    $response = array(
                        'result' => 'fail',
                        'message' => $this->language->get('447')
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (417):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('417')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }
    
    public function unblockUsers(){
        if(!empty($_POST['users'])){
            $headers = array(
                'Content-type' => 'application/json',
                'token'     => $this->session->get('token'),
            );
            $data = array(
               'userNames' => implode(',', $_POST['users']),           
                          
                          );
            $this->log('Type________'. json_encode($data), 'info');
            $result = $this->_model->unblockUsers($headers, $_POST['users']);
            $this->log('Type'. $result['body']->userElementList, 'info');
            switch ($result['status']) {
                case (200):
                    $this->log('UUUUUUUEEEEE'. json_encode($result['body']), 'info');
                    $policies = [];
                    foreach($result['body']->{'userElementList'} as $value){
                        $policy = array(
                          
                            'userName' => (string)$value->{'userName'},
                            'emailId' => (string)$value->{'emailId'},
                            'isActive' => $value->{'active'},
                            'time'        =>!empty($value->{'lastModifiedTimestamp'})? date("M j, Y, g:i:s", (((int)$value->{'lastModifiedTimestamp'}) / 1000)) : '',
                            'status' => (string)$value->{'status'},
                            
                            );
                        $policies = array_merge(array($policy), $policies);
                    }
                    
                    $this->log('UUUUUUU'. json_encode($policies), 'info');
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('deletedMultipleUsers'),
                        'naveen'  => $policies
                    );
                    echo json_encode($response);
                    exit;
                case (447):
                    $response = array(
                        'result' => 'fail',
                        'message' => $this->language->get('447')
                    );
                    echo json_encode($response);
                    exit;
                
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    /**
     * Renders users page
     */
    public function createUserGroup(){
        Auth::allowOnly('admin');
        $headers = array(
            'Content-type' => 'application/json',
            'Accept' => 'application/json',
            'token'     => $this->session->get('token')
        );
        $groupResult = $this->_model->getUserGroups($headers);
        $groups = !empty($groupResult['body']) ? json_decode($groupResult['body']) : '';

        $data = array(
            'title' => $this->language->get('indexPageName'),
            'subTitle' => $this->language->get('manageUserGroup'),
            'active' => 'users',
            'csrf_token' => $this->setCSRFToken(),
            'groups' => $groups->stringElementList,
            'loggedInUser' => $this->session->all()
        );


        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('users/userGroup', $data);
        View::renderTemplate('footer', $data);
    }



    /**
     * Below method Delete list of users
     */
    public function deleteUserGroup(){
        if(!empty($_POST['groupName'])){
            $headers = array(
                'Content-type' => 'application/json',
                'Accept' => 'application/json',
                'token'     => $this->session->get('token'),
            );

            $result = $this->_model->deleteUserGroup($headers, $_POST['groupName']);

            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('deletedAGroup'),
                    );
                    echo json_encode($response);
                    exit;
                case (447):
                    $response = array(
                        'result' => 'fail',
                        'message' => $this->language->get('447')
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (417):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('417')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','msg' => $this->language->get('noData')));
        }
    }

    function getUserXML($clouds,$users){
        $doc = new DOMDocument();
        $doc->formatOutput = true;

        $b = $doc->createElement( "user" );
        $doc->appendChild( $b );

        $userName = $doc->createElement( "user-name" );
        $userName->appendChild( $doc->createTextNode( $users['userName'] ));
        $b->appendChild( $userName );
        
        if($users['displayName'] != ''){
        $displayName = $doc->createElement( "display-name" );
        $displayName->appendChild( $doc->createTextNode( $users['displayName'] ));
        $b->appendChild( $displayName );
        }

        if($users['location'] != ''){
        $location = $doc->createElement( "location" );
        $location->appendChild( $doc->createTextNode( $users['location'] ));
        $b->appendChild( $location );
        }

        if($users['department'] != ''){
        $department = $doc->createElement( "department" );
        $department->appendChild( $doc->createTextNode( $users['department'] ));
        $b->appendChild( $department );
        }

        if($users['isBackupEnabled'] == 'true'){
        $userName = $doc->createElement( "policy-name" );
        $userName->appendChild( $doc->createTextNode( $users['policyName'] ));
        $b->appendChild( $userName );
        }

        $userName = $doc->createElement( "sync-policy-name" );
        $userName->appendChild( $doc->createTextNode( $users['syncPolicyName'] ));
        $b->appendChild( $userName );

        $password = $doc->createElement( "password-hash" );
        $password->appendChild( $doc->createTextNode( $users['password'] ));
        $b->appendChild( $password );

        $emailId = $doc->createElement( "email-id" );
        $emailId->appendChild( $doc->createTextNode( $users['emailId'] ));
        $b->appendChild( $emailId );
        
        $odbEmailId = $doc->createElement( "odb-loginId" );
        $odbEmailId->appendChild( $doc->createTextNode( $users['odb-loginId'] ));
        $b->appendChild( $odbEmailId );

        $admin = $doc->createElement( "is-admin" );
        $admin->appendChild($doc->createTextNode( $users['isAdmin'] ));
        $b->appendChild( $admin );

        $admin = $doc->createElement( "is-readonly-admin" );
        $admin->appendChild($doc->createTextNode( $users['readOnlyAdmin'] ));
        $b->appendChild( $admin );

        $sync = $doc->createElement( "sync-enabled" );
        $sync->appendChild($doc->createTextNode( $users['isSyncEnabled'] ));
        $b->appendChild( $sync );

        $backup = $doc->createElement( "backup-enabled" );
        $backup->appendChild($doc->createTextNode( $users['isBackupEnabled'] ));
        $b->appendChild( $backup );

        $serverBackup = $doc->createElement( "server-backup-enabled" );
        $serverBackup->appendChild($doc->createTextNode( $users['isServerBackupEnabled'] ));
        $b->appendChild( $serverBackup );

        $share = $doc->createElement( "share-enabled" );
        $share->appendChild($doc->createTextNode( $users['isShareEnabled'] ));
        $b->appendChild( $share );

        $mcs = $doc->createElement("mini-clouds");
        $b->appendChild($mcs);

        if(!empty($clouds)){
            foreach ($clouds as $cloud) {
                $mc= $doc->createElement("mini-cloud");
                $mcs->appendChild($mc);
                $cloudName = $doc->createElement( "mini-cloud-name" );
                $cloudName->appendChild($doc->createTextNode( $cloud));
                $mc->appendChild($cloudName);
            }
        }
        return $doc->saveXML();
    }

    public function getOsType($deviceType){
        if((strpos($deviceType, 'Wind') === false )){
            if((strpos($deviceType, 'Lin') === false )){
                return 'MAC';
            }else{
                return 'Linux';
            }
        }else{
            return 'Windows';
        }
    }

public function getAllCcS(){
        if(empty($_POST)){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
        $ccS = [];
        $headers = array(
            'Content-Type'      => 'application/json',
            'token'          => $this->session->get('token')
        );
        $result = $this->_model->getCloudCredentials($headers);
        $this->log($result);
        switch ($result['status']) {
            case (200):
                if(!empty($result['body']->{'cloudCredentialListElement'}->{'cloudCredentialElements'})){
                    foreach($result['body']->{'cloudCredentialListElement'}->{'cloudCredentialElements'} as $value){
                        switch($value->cloudName){
                            case 'azureBlob':
                                $value->cloudNameInCaps = 'Azure Blob';
                                break;
                            case 'ibmCloud':
                                $value->cloudNameInCaps = 'IBM Cloud';
                                break;
                            case 'amazon':$value->cloudNameInCaps = 'Amazon s3 Services';
                                break;
                            case 'localStorage':
                                $value->cloudNameInCaps = 'Local Storage';
                break;
                            default:
                                $value->cloudNameInCaps = '';
                                break;
                        }

                        $ccS = array_merge(array($value), $ccS);
                    }
                }
                $response = array(
                    'result' => 'success',
                    'ccS' => $ccS,
);
                echo json_encode($response);
                exit;
        case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
        case (0):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }

    public function bulkCreate(){
        if(!empty($_POST['devices'])){
            $headers = array(
                'Content-Type' => 'application/json',
                'token'     => $this->session->get('token'),
            );
            
            $this->log('Type________'. $_POST['devices'], 'info');
            $result = $this->_model->bulkCreate($headers, $_POST['devices']);
            $this->log('Type_____userElementList'. json_encode($result['body']->userElementList), 'info');
            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'msg' => "Users Created Successfully",
                    );
                    echo json_encode($response);
                    exit;
                case (201):
                        $response = array(
                            'result' => 'success',
                            'message' => $this->language->get('userSavedSuccess')
                        );
                        echo json_encode($response);
                        exit;
                case (400):
                    $users = array();
                    foreach($result['body']->userElementList as $value){
                        $user = array(
                            'userName'         => (string)$value->{'userName'},
                        );
                        $users = array_merge(array($user), $users);
                    }
                    $response = array(
                        'result' => 'Not_created',
                        'users'  => array_reverse($users),
                        'msg' => 'Unable to create users.'
                    );
                    echo json_encode($response);
                    exit;
                case (447):
                    $response = array(
                        'result' => 'fail',
                        'message' => $this->language->get('447')
                    );
                    echo json_encode($response);
                    exit;
                case (417):
                    $result = array(
                        'result' => 'User_Limit_Exceeded',
                        'noOfPossibleUsers' => $result['header']->{'container'}['no-of-possible-users'],
                        'msg' => $this->language->get('417')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    public function getOsIcon($deviceType){
        if((strpos($deviceType, 'Wind') === false )){
            if((strpos($deviceType, 'Lin') === false )){
                if(strpos($deviceType, 'Android') === false){
                    $type =  'apple';
                }else{
                    $type =  'android';
                }
            }else{
                $type =  'linux';
            }
        }else{
            $type =  'windows';
        }
        return $type;
    }

}
