<?php

namespace Controllers;

use Core\Language;
use Core\View;
use Core\Controller;
use Helpers\Url;
use Helpers\Auth;
use Models\Device;
/*
 * Copyright 2014
 * ParaBlu Systems Private Limited
 * All Rights Reserved
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of ParaBlu Systems Private Limited and its suppliers, if any.
 * The intellectual and technical concepts contained herein are proprietary
 * to ParaBlu Systems Private Limited and its suppliers and may be covered by
 * Indian, US and Foreign Patents, patents in process, and are protected by
 * trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from ParaBlu Systems Private Limited.
 *
 * Login controller
 *
 * @author Prashanth Kumar B - prashanth.kumar@parablu.com
 * @version 2.0
 * @date June 10, 2015
 *
 */

class DeviceController extends Controller{

    /**
     * Below variable contains the object reference of a model
     */

    protected $_model;

    /**
     * Call the parent construct
     */
    public function __construct(){
        parent::__construct();
        $this->auth->loginCheck();
        $this->language->load('device');
        $this->_model = new Device();
    }

    public function getDevicePolicyInformation($userName, $deviceUUID, $os){
        $this->auth->checkLoggedInUser($userName);
        $this->session->put('currentDeviceUUID', $deviceUUID);
        $this->session->put('currentOs', $os);
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('device'),
            'crumb2' => $this->language->get('policy_info'),
            'active' => 'users',
            'os' => $os,
            'deviceName' => htmlentities($_GET['deviceName']),
            'currentUser' => $userName,
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('device/policyInfo', $data);
        View::renderTemplate('footer', $data);
    }

    public function getCurrentDeviceIDInformation(){
        if(!empty($_POST) && !empty($this->session->get('currentDeviceUUID'))){
            $headers = array(
                'token' => $this->session->get('token'),
                'deviceUUID' => $this->session->get('currentDeviceUUID')
            );
            $result = $this->_model->userDevicePolicyInfo($headers);
            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'defaultFolders' => array_map('strtolower', $result['body']->bluSyncPortalBackupPolicyElement->defaultFolders),
                        'userExcludedFolders' => array_map('strtolower', $result['body']->bluSyncPortalBackupPolicyElement->userExcludedFolders),
                        'userDefinedFolders' => array_map('strtolower', $result['body']->bluSyncPortalBackupPolicyElement->userDefinedFolders),
                        'fileTypes' => $result['body']->bluSyncPortalBackupPolicyElement->fileTypes,
			'filterType' => $result['body']->bluSyncPortalBackupPolicyElement->filterType,
                        'maxSizeAllowed' => round(($result['body']->bluSyncPortalBackupPolicyElement->maxSizeAllowed) / 1024,2)
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function addInclusionFolder(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST) && !empty($this->session->get('currentDeviceUUID'))){
            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token')
            );
            $data = json_encode(
                array(
                    'folderPath' => $_POST['folder'],
                    'deviceUUID' => $this->session->get('currentDeviceUUID'),
                    'userName' => $this->session->get('userName')
                )
            );
            $result = $this->_model->addInclusionFolder($headers, $data);
            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('added_inclusion_folder'),
                        'data' => $result
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (447):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('447')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'msg' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function removeInclusionFolder(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST) && !empty($this->session->get('currentDeviceUUID'))){

            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token'),
                'folderPath' => $_POST['folder'],
                'deviceUUID' => $this->session->get('currentDeviceUUID'),
                'userName' => $this->session->get('userName')
            );

            $result = $this->_model->removeInclusionFolder($headers);

            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('removed_inclusion_folder'),
                        'data' => $result
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (447):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('447')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'msg' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function addExclusionFolder(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST) && !empty($this->session->get('currentDeviceUUID'))){
            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token')
            );
            $data = json_encode(
                array(
                    'folderPath' => $_POST['folder'],
                    'deviceUUID' => $this->session->get('currentDeviceUUID'),
                    'userName' => $this->session->get('userName')
                )
            );
            $result = $this->_model->addExclusionFolder($headers, $data);
            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('added_exclusion_folder'),
                        'data' => $result
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (447):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('447')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'msg' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function removeExclusionFolder(){
        $this->verifyAndCleanItUp();
        if(!empty($_POST) && !empty($this->session->get('currentDeviceUUID'))){

            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token'),
                'folderPath' => $_POST['folder'],
                'deviceUUID' => $this->session->get('currentDeviceUUID'),
                'userName' => $this->session->get('userName')
            );

            $result = $this->_model->removeExclusionFolder($headers);

            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('removed_exclusion_folder'),
                        'data' => $result
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (447):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('447')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'msg' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function devicesBulkBlock(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('devices'),
            'crumb1' => $this->language->get('devicesBulkBlock'),
            'active' => 'devices',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('device/devicesBulkBlock', $data);
        View::renderTemplate('footer', $data);
    }

    public function devices(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('devices'),
            'crumb1' => $this->language->get('listOfDevices'),
            'active' => 'devices',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('device/devices', $data);
        View::renderTemplate('footer', $data);
    }

    public function getAllDevices(){
        Auth::allowOnly('admin');
        $this->verifyAndCleanItUp();
        if(!empty($_POST)){

            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token'),
                'skipValue' => $_POST['skipValue']
            );

            if($_POST['deviceName'] != 'all'){
                $headers['deviceNames'] = $_POST['deviceName'];
                $result = $this->_model->searchInDevices($headers);
            }else{
                $result = $this->_model->getAllDevices($headers);
            }
            $this->log('Result: '.json_encode($result), 'info');
            switch ($result['status']) {
                case (200):

                    $devices = [];

                    if(!empty($result['body'])){
                                                foreach($result['body']->{'device-info'} as $value){

                            $version = (string)$value->{'device-os-type'};
                            if( (preg_match('/Win/', (string)$value->{'device-os-type'} )) && (preg_match('/bit/', (string)$value->{'device-os-version'} ))) {
                                     $char = "(";

                                $str = explode($char,(string)$value->{'device-os-version'});
                                $version =$version."(".$str[1];
}
                            $device = array(
                                'profilePic' => Url::profilePic((string)$value->{'userName'}),
                                'decoupled' => (string)$value->{'decoupled'},
                                'deviceId' => (string)$value->{'device-id'},
                                'createdTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'device-created-timestamp'}) / 1000)),
                                'modifiedTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'modified-timestamp'}) / 1000)),
                                'modifiedTimeStampUnix' => (int)$value->{'modified-timestamp'},
                                'deviceName' => (string)$value->{'device-name'},
                                'icon' => $this->getOsIcon($value->{'device-os-type'}),
                                'version' => (string)$value->{'device-os-version'},
                                'tempPath' => (string)$value->{'device-temp-path'},
                                'deviceUUID' => (string)$value->{'device-uuid'},
                                'blocked' => (string)$value->{'is-blocked'},
                                'userName' => (string)$value->{'userName'},
                                'osType' =>  $version

                            );
                            $devices = array_merge(array($device), $devices);
                        }
                    }


                    $response = array(
                        'result' => 'success',
                        'devices' => $devices
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function blockUnBlockDevice(){

        Auth::allowOnly('admin');
        $this->verifyAndCleanItUp();
        if(!empty($_POST)){

            $headers = array(
                'Content-Type' => 'application/xml',
                'Accept' => 'application/xml',
                'Accept-Charset' => 'utf-8',
                'token' => $this->session->get('token'),
            );

            $device = array(
                'willRemove'                => $_POST['device']['decoupled'],
                'decoupled'                 => $_POST['device']['decoupled'],
                'is-blocked'                => $_POST['action'],
                'device-created-timestamp'  => $_POST['device']['createdTimeStamp'],
                'device-name'               => $_POST['device']['deviceName'],
                'userName'                  => $_POST['device']['userName'],
                'device-temp-path'          => $_POST['device']['tempPath'],
                'device-uuid'               => $_POST['device']['deviceUUID'],
                'modified-timestamp'        => $_POST['device']['modifiedTimeStampUnix'],
                'device-os-type'            => $_POST['device']['osType'],
                'device-os-version'         => $_POST['device']['version']
            );

            $device = array_flip($device);
            $xml = new \SimpleXMLElement('<?xml version="1.0" encoding="UTF-8" standalone="yes"?><device-info/>');
            array_walk_recursive($device, array ($xml, 'addChild'));
            $device = $xml->asXML();

            $result = $this->_model->blockUnBlockDevice($headers, $this->session->get('userName'), $device);

            switch ($result['status']) {
                case (200):

                    if($_POST['action'] == 'true'){
                        $msg = $this->language->get('deviceBlocked');
                    }else{
                        $msg = $this->language->get('deviceUnBlocked');
                    }

                    $response = array(
                        'result' => 'success',
                        'msg' => $msg
                    );
                    echo json_encode($response);
                    exit;

                case (418):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('deviceLimitReached')
                    );
                    echo json_encode($result);
                    exit();

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function deleteDevice(){

        Auth::allowOnly('admin');
        $this->verifyAndCleanItUp();
        if(!empty($_POST)){

            $headers = array(
                'Content-Type' => 'application/xml',
                'Accept' => 'application/xml',
                'Accept-Charset' => 'utf-8',
                'token' => $this->session->get('token'),
            );

            $device = array(
                'decoupled'                 => $_POST['device']['decoupled'],
                'is-deleted'                => $_POST['device']['blocked'],
                'device-created-timestamp'  => $_POST['device']['createdTimeStamp'],
                'device-name'               => $_POST['device']['deviceName'],
                'userName'                  => $_POST['device']['userName'],
                'device-temp-path'          => $_POST['device']['tempPath'],
                'device-uuid'               => $_POST['device']['deviceUUID'],
                'modified-timestamp'        => $_POST['device']['modifiedTimeStampUnix'],
                'device-os-type'            => $_POST['device']['osType'],
                'device-os-version'         => $_POST['device']['version']
            );
            $deviceSecond = array(
                'is-blocked'                => $_POST['device']['blocked'],
            );
            $device = array_flip($device);
            $deviceSecond = array_flip($deviceSecond);
            $xml = new \SimpleXMLElement('<?xml version="1.0" encoding="UTF-8" standalone="yes"?><device-info/>');
            array_walk_recursive($device, array ($xml, 'addChild'));
            array_walk_recursive($deviceSecond, array ($xml, 'addChild'));
            $device = $xml->asXML();

            $result = $this->_model->deleteDevice($headers, $this->session->get('userName'), $device);

            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('deviceDeleted')
                    );
                    echo json_encode($response);
                    exit;

                case (418):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('deviceLimitReachedDelete')
                    );
                    echo json_encode($result);
                    exit();

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function blockMultipleDevices(){

        Auth::allowOnly('admin');
        $this->verifyAndCleanItUp();

        if(!empty($_POST)){

            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token'),
                'active' => $_POST['blocked'],
            );

            $result = $this->_model->blockUnBlockMultipleDevice($headers, $_POST['deviceUUIDs']);

            switch ($result['status']) {
                case (200):

                    if($_POST['blocked'] == 'true'){
                        $msg = $this->language->get('multipleDeviceBlocked');
                    }else{
                        $msg = $this->language->get('multipleDeviceUnBlocked');
                    }

                    $response = array(
                        'result' => 'success',
                        'msg' => $msg
                    );
                    echo json_encode($response);
                    exit;

                case (418):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('deviceLimitReached')
                    );
                    echo json_encode($result);
                    exit();

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }
    
    public function blockDevices(){
        if(!empty($_POST['devices'])){
            $headers = array(
                'Content-Type' => 'application/json',
                'token'     => $this->session->get('token'),
            );
            
            $this->log('Type________'. $_POST['devices'], 'info');
            $result = $this->_model->blockDevices($headers, $_POST['devices']);
            $this->log('Type'. $result['body']->deviceElementList, 'info');
            switch ($result['status']) {
                case (200):
                    //$this->log('UUUUUUUEEEEE'. json_encode($result['body']), 'info');
                    $policies = [];
                    foreach($result['body']->{'deviceElementList'} as $value){
                        $policy = array(
                          
                            'userName' => (string)$value->{'userName'},
                            'deviceName' => (string)$value->{'deviceName'},
                            'deviceUUID' => (string)$value->{'deviceUUID'},
                            'time'        => date("M j, Y, g:i:s", (((int)$value->{'lastModifiedTimestamp'}) / 1000)),
                            'status' => (string)$value->{'status'},
                            );
                        $policies = array_merge(array($policy), $policies);
                    }
                    
                    $this->log('UUUUUUU'. json_encode($policies), 'info');
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('deletedMultipleUsers'),
                        'naveen'  => array_reverse($policies)
                    );
                    echo json_encode($response);
                    exit;
                case (447):
                    $response = array(
                        'result' => 'fail',
                        'message' => $this->language->get('447')
                    );
                    echo json_encode($response);
                    exit;
                
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }
        }else{
            echo json_encode(array('result' => 'noData','message' => $this->language->get('noData')));
        }
    }

    public function deleteMultipleDevices(){

        Auth::allowOnly('admin');
        $this->verifyAndCleanItUp();

        if(!empty($_POST)){

            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token'),
                'deletedBy'    => $this->session->get('userName')
            );

            $result = $this->_model->deleteMultipleDevices($headers, $_POST['deviceUUIDs']);

            switch ($result['status']) {
                case (200):
                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('multipleDevicesDeleted')
                    );
                    echo json_encode($response);
                    exit;

                case (418):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('deviceLimitReached')
                    );
                    echo json_encode($result);
                    exit();

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }
    public function searchDevices(){
        if(!empty($_GET)){

            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token'),
                'skipValue' => $_GET['skipValue']
            );

            if(!empty($_GET['q']) && $_GET['q'] != 'all'){
                $headers['deviceNames'] = $_GET['q'];
                $result = $this->_model->searchInDevices($headers);
            }else{
                $result = $this->_model->getAllDevices($headers);
            }

            switch ($result['status']) {
                case (200):

                    $devices = [];
                    if(!empty($result['body'])){
                        foreach($result['body']->{'device-info'} as $value){
                            $device = array(
                                'profilePic' => Url::profilePic((string)$value->{'userName'}),
                                'decoupled' => (string)$value->{'decoupled'},
                                'createdTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'device-created-timestamp'}) / 1000)),
                                'modifiedTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'modified-timestamp'}) / 1000)),
                                'modifiedTimeStampUnix' => (int)$value->{'modified-timestamp'},
                                'deviceName' => (string)$value->{'device-name'},
                                'osType' => (string)$value->{'device-os-type'},
                                'icon' => (strpos((string)$value->{'device-os-type'},'Win') !== false ? 'windows' : 'apple'),
                                'version' => (string)$value->{'device-os-version'},
                                'tempPath' => (string)$value->{'device-temp-path'},
                                'deviceUUID' => (string)$value->{'device-uuid'},
                                'blocked' => (string)$value->{'is-blocked'},
                                'userName' => (string)$value->{'userName'},
                            );
                            $devices = array_merge(array($device), $devices);
                        }
                    }
                    $devices = array_merge(array(array("deviceName" => "all", "profilePic" => Url::profilePic("default"))), $devices);
                    $response = array(
                        'result' => 'success',
                        'devices' => $devices
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function getAllBlockedDevices(){
        if(empty($_GET)){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
            exit;
        }

        $headers = array(
            'Content-Type' => 'application/json',
            'token' => $this->session->get('token'),
            'skipValue' => $_GET['skipValue']
        );

        if(!empty($_GET['q']) && $_GET['q'] != 'all'){
            $headers['deviceNames'] = $_GET['q'];
            $result = $this->_model->searchInDevices($headers);
        }else{
            $result = $this->_model->getAllDevices($headers);
        }

        switch ($result['status']) {
            case (200):

                $devices = [];

                if(!empty($result['body'])){
                    foreach($result['body']->{'device-info'} as $value){
                        if((string)$value->{'is-blocked'} == 'true'){
                            $device = array(
                                'profilePic' => Url::profilePic((string)$value->{'userName'}),
                                'decoupled' => (string)$value->{'decoupled'},
                                'createdTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'device-created-timestamp'}) / 1000)),
                                'modifiedTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'modified-timestamp'}) / 1000)),
                                'modifiedTimeStampUnix' => (int)$value->{'modified-timestamp'},
                                'deviceName' => (string)$value->{'device-name'},
                                'osType' => (string)$value->{'device-os-type'},
                                'icon' => (strpos((string)$value->{'device-os-type'},'Win') !== false ? 'windows' : 'apple'),
                                'version' => (string)$value->{'device-os-version'},
                                'tempPath' => (string)$value->{'device-temp-path'},
                                'deviceUUID' => (string)$value->{'device-uuid'},
                                'blocked' => (string)$value->{'is-blocked'},
                                'userName' => (string)$value->{'userName'},
                            );
                            $devices = array_merge(array($device), $devices);
                        }
                    }
                }

                $response = array(
                    'result' => 'success',
                    'devices' => $devices
                );
                echo json_encode($response);
                exit;
            case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
            case (0):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }

    public function getAllUnBlockedDevices(){
        if(empty($_GET)){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
            exit;
        }
        $headers = array(
            'Content-Type' => 'application/json',
            'token' => $this->session->get('token'),
            'skipValue' => $_GET['skipValue']
        );

        if(!empty($_GET['q']) && $_GET['q'] != 'all'){
            $headers['deviceNames'] = $_GET['q'];
            $result = $this->_model->searchInDevices($headers);
        }else{
            $result = $this->_model->getAllDevices($headers);
        }

        switch ($result['status']) {
            case (200):

                $devices = [];

                if(!empty($result['body'])){
                    foreach($result['body']->{'device-info'} as $value){
                        if((string)$value->{'is-blocked'} == 'false'){
                            $device = array(
                                'profilePic' => Url::profilePic((string)$value->{'userName'}),
                                'decoupled' => (string)$value->{'decoupled'},
                                'createdTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'device-created-timestamp'}) / 1000)),
                                'modifiedTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'modified-timestamp'}) / 1000)),
                                'modifiedTimeStampUnix' => (int)$value->{'modified-timestamp'},
                                'deviceName' => (string)$value->{'device-name'},
                                'osType' => (string)$value->{'device-os-type'},
                                'icon' => (strpos((string)$value->{'device-os-type'},'Win') !== false ? 'windows' : 'apple'),
                                'version' => (string)$value->{'device-os-version'},
                                'tempPath' => (string)$value->{'device-temp-path'},
                                'deviceUUID' => (string)$value->{'device-uuid'},
                                'blocked' => (string)$value->{'is-blocked'},
                                'userName' => (string)$value->{'userName'},
                            );
                            $devices = array_merge(array($device), $devices);
                        }
                    }
                }

                $response = array(
                    'result' => 'success',
                    'devices' => $devices
                );
                echo json_encode($response);
                exit;
            case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
            case (0):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }

    public function getOsType($deviceType){
        if((strpos($deviceType, 'Wind') === false )){
            if((strpos($deviceType, 'Lin') === false )){
                return 'MAC';
            }else{
                return 'Linux';
            }
        }else{
            return 'Windows';
        }
    }

    public function getOsIcon($deviceType){
        if((strpos($deviceType, 'Wind') === false )){
            if((strpos($deviceType, 'Lin') === false )){
                if(strpos($deviceType, 'Android') === false){
                    $type =  'apple';
                }else{
                    $type =  'android';
                }
            }else{
                $type =  'linux';
            }
        }else{
            $type =  'windows';
        }
        return $type;
    }

    public function eventHub(){

            $headers = array(
                'token' => $this->session->get('token')
            );
            $result = $this->_model->getEventHub($headers, $_POST['deviceId']);

            switch ($result['status']) {
                case (200):
                    $response = array(
                        'actionToDeviceUUID' => $result['body']->{'eventHubElement'}->{'actionToDeviceUUID'},
                        'eventHub' => $result['body']->{'eventHubElement'}->{'actions'},
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
            case (0):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }
    
    public function saveEventHub(){
        
                $headers = array(
                    'Content-Type' => 'application/json',
                    'token'         => $this->session->get('token')
                );

            $eventHubElements = [];
                    /*if(isset($_POST['startBackup']) && $_POST['startBackup'] == 'Yes' ){
                        array_push($eventHubElements,'START_BACKUP');
                        }
                    if(isset($_POST['stopBackup']) && $_POST['stopBackup'] == 'Yes'){
                        array_push($eventHubElements,'STOP_BACKUP');
                        }
                    if(isset($_POST['backupPolicyRefresh']) && $_POST['backupPolicyRefresh'] == 'Yes'){
                        array_push($eventHubElements,'BACKUPPOLICY_REFRESH');
                        }*/
                    if(isset($_POST['agentLogs']) && $_POST['agentLogs'] == 'Yes'){
                        array_push($eventHubElements,'PULL_AGENTLOGS');
                        }

                $data = array(
                    "actionToDeviceUUID" => $_POST['actionToDeviceUUID'],
                    "actions"         => $eventHubElements,
                );
                
                $result = $this->_model->saveEventHub($headers, $data);
                switch ($result['status']) {
                    case (200):
                        $response = array(
                            'result' => 'success',
                            'msg' => "Settings Saved"
                        );
                        echo json_encode($response);
                        exit;
                    case (421):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('421')
                        );
                        echo json_encode($result);
                        exit();
                    case (500):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('500')
                        );
                        echo json_encode($result);
                        exit();
                    case (409):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('policyExists')
                        );
                        echo json_encode($result);
                        exit();

                    case (0):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('0')
                        );
                        echo json_encode($result);
                        exit();
                    default:
                        $result = array(
                            'result' => 'fail',
                            'code' => $result,
                            'msg' => $this->language->get('default')
                        );
                        echo json_encode($result);
                        exit();
                }
            
    }
}
