/*
 * Decompiled with CFR 0.152.
 */
package org.jclouds.ec2.compute.functions;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.base.Strings;
import com.google.common.base.Supplier;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableCollection;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.google.inject.Inject;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import javax.annotation.Resource;
import javax.inject.Singleton;
import org.jclouds.collect.Memoized;
import org.jclouds.compute.domain.Hardware;
import org.jclouds.compute.domain.HardwareBuilder;
import org.jclouds.compute.domain.Image;
import org.jclouds.compute.domain.NodeMetadata;
import org.jclouds.compute.domain.NodeMetadataBuilder;
import org.jclouds.compute.domain.Volume;
import org.jclouds.compute.domain.internal.VolumeImpl;
import org.jclouds.compute.functions.GroupNamingConvention;
import org.jclouds.compute.util.ComputeServiceUtils;
import org.jclouds.domain.Credentials;
import org.jclouds.domain.Location;
import org.jclouds.domain.LoginCredentials;
import org.jclouds.ec2.compute.domain.RegionAndName;
import org.jclouds.ec2.domain.BlockDevice;
import org.jclouds.ec2.domain.InstanceState;
import org.jclouds.ec2.domain.RootDeviceType;
import org.jclouds.ec2.domain.RunningInstance;
import org.jclouds.logging.Logger;
import org.jclouds.util.InetAddresses2;

@Singleton
public class RunningInstanceToNodeMetadata
implements Function<RunningInstance, NodeMetadata> {
    @Resource
    protected Logger logger = Logger.NULL;
    protected final Supplier<Set<? extends Location>> locations;
    protected final Supplier<Set<? extends Hardware>> hardware;
    protected final Supplier<LoadingCache<RegionAndName, ? extends Image>> imageMap;
    protected final Map<String, Credentials> credentialStore;
    protected final Map<InstanceState, NodeMetadata.Status> instanceToNodeStatus;
    protected final GroupNamingConvention.Factory namingConvention;

    @Inject
    protected RunningInstanceToNodeMetadata(Map<InstanceState, NodeMetadata.Status> instanceToNodeStatus, Map<String, Credentials> credentialStore, Supplier<LoadingCache<RegionAndName, ? extends Image>> imageMap, @Memoized Supplier<Set<? extends Location>> locations, @Memoized Supplier<Set<? extends Hardware>> hardware, GroupNamingConvention.Factory namingConvention) {
        this.locations = Preconditions.checkNotNull(locations, "locations");
        this.hardware = Preconditions.checkNotNull(hardware, "hardware");
        this.imageMap = Preconditions.checkNotNull(imageMap, "imageMap");
        this.instanceToNodeStatus = Preconditions.checkNotNull(instanceToNodeStatus, "instanceToNodeStatus");
        this.credentialStore = Preconditions.checkNotNull(credentialStore, "credentialStore");
        this.namingConvention = Preconditions.checkNotNull(namingConvention, "namingConvention");
    }

    @Override
    public NodeMetadata apply(RunningInstance instance) {
        if (instance == null || instance.getId() == null) {
            return null;
        }
        NodeMetadataBuilder builder = new NodeMetadataBuilder();
        builder.name(instance.getTags().get("Name"));
        ComputeServiceUtils.addMetadataAndParseTagsFromValuesOfEmptyString(builder, instance.getTags());
        builder.providerId(instance.getId());
        builder.id(instance.getRegion() + "/" + instance.getId());
        String group = this.getGroupForInstance(instance);
        builder.group(group);
        if (instance.getPrivateDnsName() != null) {
            builder.hostname(instance.getPrivateDnsName().replaceAll("\\..*", ""));
        }
        this.addCredentialsForInstance(builder, instance);
        builder.status(this.instanceToNodeStatus.get((Object)instance.getInstanceState()));
        builder.backendStatus(instance.getRawState());
        ImmutableSet.Builder addressesBuilder = ImmutableSet.builder();
        if (Strings.emptyToNull(instance.getIpAddress()) != null) {
            addressesBuilder.add(instance.getIpAddress());
        }
        if (Strings.emptyToNull(instance.getPrivateIpAddress()) != null) {
            addressesBuilder.add(instance.getPrivateIpAddress());
        }
        ImmutableCollection addresses = addressesBuilder.build();
        builder.publicAddresses(Iterables.filter(addresses, Predicates.not(InetAddresses2.IsPrivateIPAddress.INSTANCE)));
        builder.privateAddresses(Iterables.filter(addresses, InetAddresses2.IsPrivateIPAddress.INSTANCE));
        builder.hardware(this.parseHardware(instance));
        Location location = this.getLocationForAvailabilityZoneOrRegion(instance);
        builder.location(location);
        builder.imageId(instance.getRegion() + "/" + instance.getImageId());
        RegionAndName regionAndName = new RegionAndName(instance.getRegion(), instance.getImageId());
        try {
            Image image = this.imageMap.get().getUnchecked(regionAndName);
            if (image != null) {
                builder.operatingSystem(image.getOperatingSystem());
            }
        }
        catch (CacheLoader.InvalidCacheLoadException e) {
            this.logger.debug("image not found for %s: %s", regionAndName, e);
        }
        catch (UncheckedExecutionException e) {
            this.logger.debug("error getting image for %s: %s", regionAndName, e);
        }
        return builder.build();
    }

    protected void addCredentialsForInstance(NodeMetadataBuilder builder, RunningInstance instance) {
        builder.credentials(LoginCredentials.fromCredentials(this.credentialStore.get("node#" + instance.getRegion() + "/" + instance.getId())));
    }

    protected Hardware parseHardware(RunningInstance instance) {
        Hardware hardware = this.getHardwareForInstance(instance);
        if (hardware != null) {
            hardware = HardwareBuilder.fromHardware(hardware).volumes(RunningInstanceToNodeMetadata.addEBS(instance, hardware.getVolumes())).build();
        }
        return hardware;
    }

    @VisibleForTesting
    static List<Volume> addEBS(final RunningInstance instance, Iterable<? extends Volume> volumes) {
        Iterable<Volume> ebsVolumes = Iterables.transform(instance.getEbsBlockDevices().entrySet(), new Function<Map.Entry<String, BlockDevice>, Volume>(){

            @Override
            public Volume apply(Map.Entry<String, BlockDevice> from) {
                return new VolumeImpl(from.getValue().getVolumeId(), Volume.Type.SAN, null, from.getKey(), instance.getRootDeviceName() != null && instance.getRootDeviceName().equals(from.getKey()), true);
            }
        });
        if (instance.getRootDeviceType() == RootDeviceType.EBS) {
            volumes = Iterables.filter(volumes, new Predicate<Volume>(){

                @Override
                public boolean apply(Volume input) {
                    return !input.isBootDevice();
                }
            });
        }
        return Lists.newArrayList(Iterables.concat(volumes, ebsVolumes));
    }

    @VisibleForTesting
    String getGroupForInstance(RunningInstance instance) {
        String group = this.parseGroupFrom(instance, instance.getGroupNames());
        if (group == null && instance.getKeyName() != null) {
            group = this.parseGroupFrom(instance, Sets.newHashSet(instance.getKeyName()));
        }
        return group;
    }

    private String parseGroupFrom(RunningInstance instance, Set<String> data) {
        String group = null;
        try {
            Predicate<String> containsAnyGroup = this.namingConvention.create().containsAnyGroup();
            String encodedGroup = Iterables.getOnlyElement(Iterables.filter(data, containsAnyGroup));
            group = this.namingConvention.create().extractGroup(encodedGroup);
        }
        catch (NoSuchElementException e) {
            this.logger.debug("no group parsed from %s's data: %s", instance.getId(), data);
        }
        catch (IllegalArgumentException e) {
            this.logger.debug("too many groups match naming convention; %s's data: %s", instance.getId(), data);
        }
        return group;
    }

    @VisibleForTesting
    Hardware getHardwareForInstance(final RunningInstance instance) {
        try {
            return Iterables.find((Iterable)this.hardware.get(), new Predicate<Hardware>(){

                @Override
                public boolean apply(Hardware input) {
                    return input.getId().equals(instance.getInstanceType());
                }
            });
        }
        catch (NoSuchElementException e) {
            this.logger.debug("couldn't match instance type %s in: %s", instance.getInstanceType(), this.hardware.get());
            return null;
        }
    }

    private Location getLocationForAvailabilityZoneOrRegion(RunningInstance instance) {
        Location location = this.findLocationWithId(instance.getAvailabilityZone());
        if (location == null) {
            location = this.findLocationWithId(instance.getRegion());
        }
        return location;
    }

    private Location findLocationWithId(final String locationId) {
        if (locationId == null) {
            return null;
        }
        try {
            Location location = Iterables.find((Iterable)this.locations.get(), new Predicate<Location>(){

                @Override
                public boolean apply(Location input) {
                    return input.getId().equals(locationId);
                }
            });
            return location;
        }
        catch (NoSuchElementException e) {
            this.logger.debug("couldn't match instance location %s in: %s", locationId, this.locations.get());
            return null;
        }
    }
}

