<?php

namespace Controllers;

use Core\Language;
use Core\View;
use Core\Controller;
use Helpers\Auth;
use Helpers\Url;
use Models\Pms;
/*
 * Copyright 2014
 * ParaBlu Systems Private Limited
 * All Rights Reserved
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of ParaBlu Systems Private Limited and its suppliers, if any.
 * The intellectual and technical concepts contained herein are proprietary
 * to ParaBlu Systems Private Limited and its suppliers and may be covered by
 * Indian, US and Foreign Patents, patents in process, and are protected by
 * trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from ParaBlu Systems Private Limited.
 *
 * Login controller
 *
 * @author Prashanth Kumar B - prashanth.kumar@parablu.com
 * @version 2.0
 * @date June 10, 2015
 *
 */

class PmsController extends Controller{

    /**
     * Below variable contains the object reference of a model
     */

    protected $_model;

    /**
     * Call the parent construct
     */
    public function __construct(){
        parent::__construct();
        $this->auth->loginCheck();
        $this->language->load('pms');
        $this->_model = new Pms();
        $this->verifyAndCleanItUp();
    }

    public function pms(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'active' => 'pms',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/pms', $data);
        View::renderTemplate('footer', $data);
    }

    public function schedules(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('schedules'),
            'active' => 'pms',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/schedule/view', $data);
        View::renderTemplate('footer', $data);
    }

    public function getSchedules(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'token' => $this->session->get('token')
            );
            $result = $this->_model->getSchedules($headers);
            switch ($result['status']) {
                case (200):
                    $schedules = [];

                    if($result['body']->{'scheduleElementList'}){
                        foreach($result['body']->{'scheduleElementList'} as $value){
                            $schedule = array(
                                'time' => (string)$value->{'time'},
                                'timeZone' => (string)$value->{'timeZone'},
                                'scheduleName' => (string)$value->{'scheduleName'},
                                'daysArray' => (array)$value->{'dayArray'},
                                'backupType' => (string)$value->{'backupType'},
                                'days' => $this->getDaysString(implode(',', (array)$value->{'dayArray'})),
                            );
                            $schedules = array_merge(array($schedule), $schedules);
                        }
                    }

                    $response = array(
                        'result' => 'success',
                        'schedules' => $schedules
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function getSchedule(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'Content-Type' => 'application/json',
                'token' => $this->session->get('token'),
                'schedules' => implode(',', $_POST['scheduleNames'])
            );
            $result = $this->_model->getSchedule($headers);
            switch ($result['status']) {
                case (200):
                    $schedules = [];

                    if($result['body']->{'scheduleElementList'}){
                        foreach($result['body']->{'scheduleElementList'} as $value){
                            $schedule = array(
                                'time' => (string)$value->{'time'},
                                'timeZone' => (string)$value->{'timeZone'},
                                'scheduleName' => (string)$value->{'scheduleName'},
                                'daysArray' => (array)$value->{'dayArray'},
                                'backupType' => (string)$value->{'backupType'},
                                'days' => $this->getDaysString(implode(',', (array)$value->{'dayArray'})),
                            );
                            $schedules = array_merge(array($schedule), $schedules);
                        }
                    }

                    $response = array(
                        'result' => 'success',
                        'schedules' => $schedules
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function createSchedule(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('schedule'),
            'crumb2' => $this->language->get('create'),
            'active' => 'pms',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/schedule/create', $data);
        View::renderTemplate('footer', $data);
    }

    public function saveSchedule(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){

            $rules = array(
                'scheduleName'       => 'required',
                'scheduleStartTime'  => 'required',
                'days'               => 'required'
            );
            $filters = array(
                'scheduleName'       => 'trim|sanitize_string',
                'scheduleStartTime'  => 'trim|sanitize_string'
            );
            $filteredData = $this->valid->filter($_POST, $filters);
            $validate = $this->valid->validate($filteredData, $rules);

            if($validate == true){
                $headers = array(
                    'Content-Type'         => 'application/json',
                    'token'         => $this->session->get('token')
                );

                $data = array(
                    'scheduleName' => $_POST['scheduleName'],
                    'time' => $_POST['scheduleStartTime'],
                    'timeZone' => $_POST['scheduleTimeZone'],
                    'backupType' => $_POST['backupType'],
                    'dayArray' => $_POST['days']
                );

                $result = $this->_model->saveSchedule($headers, json_encode($data));
                switch ($result['status']) {
                    case (200):
                        $response = array(
                            'result' => 'success',
                            'msg' => $this->language->get('scheduleSaved')
                        );
                        echo json_encode($response);
                        exit;
                    case (421):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('421')
                        );
                        echo json_encode($result);
                        exit();
                    case (500):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('500')
                        );
                        echo json_encode($result);
                        exit();
                    case (0):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('0')
                        );
                        echo json_encode($result);
                        exit();
                    default:
                        $result = array(
                            'result' => 'fail',
                            'code' => $result,
                            'msg' => $this->language->get('default')
                        );
                        echo json_encode($result);
                        exit();
                }
            }else{
                $result = array(
                    'result' => 'fieldFail',
                    'msg' => $this->valid->get_errors_array()
                );
                echo json_encode($result);
            }
        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function getDaysString($numString){
        $numString = str_replace('1', Language::show('Sunday', 'keywords'), $numString);
        $numString = str_replace('2', Language::show('Monday', 'keywords'), $numString);
        $numString = str_replace('3', Language::show('Tuesday', 'keywords'), $numString);
        $numString = str_replace('4', Language::show('Wednesday', 'keywords'), $numString);
        $numString = str_replace('5', Language::show('Thursday', 'keywords'), $numString);
        $numString = str_replace('6', Language::show('Friday', 'keywords'), $numString);
        return str_replace('7', Language::show('Saturday', 'keywords'), $numString);
    }

    public function filters(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('filters'),
            'active' => 'pms',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/filter/view', $data);
        View::renderTemplate('footer', $data);
    }

    public function createFilter(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('filters'),
            'crumb2' => $this->language->get('create'),
            'active' => 'pms',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/filter/create', $data);
        View::renderTemplate('footer', $data);
    }

    public function getFilters(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'token' => $this->session->get('token')
            );
            $result = $this->_model->getFilters($headers);
            switch ($result['status']) {
                case (200):
                    $filters = [];

                    if($result['body']->{'inclusionFilterElementList'}){
                        foreach($result['body']->{'inclusionFilterElementList'} as $value){
                            $filter = array(
                                'filters' => (string)$value->{'filters'},
                                'filterName' => (string)$value->{'filterName'},
                                'filterType' => (string)$value->{'filterType'},
                                'filterArray' => (array)explode(',', (string)$value->{'filters'})
                            );
                            $filters = array_merge(array($filter), $filters);
                        }
                    }

                    $response = array(
                        'result' => 'success',
                        'filters' => $filters
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function getFilter(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'token' => $this->session->get('token')
            );
            $result = $this->_model->getFilter($headers, $_POST['filterName']);
            switch ($result['status']) {
                case (200):
                    $filters = [];

                    if($result['body']->{'inclusionFilterElementList'}){
                        foreach($result['body']->{'inclusionFilterElementList'} as $value){
                            $filter = array(
                                'filters' => (string)$value->{'filters'},
                                'filterName' => (string)$value->{'filterName'},
                                'filterType' => (string)$value->{'filterType'},
                                'filterArray' => (array)explode(',', (string)$value->{'filters'})
                            );
                            $filters = array_merge(array($filter), $filters);
                        }
                    }

                    $response = array(
                        'result' => 'success',
                        'filters' => (array)$result['body']->{'inclusionFilterElement'}
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function saveFilter(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){

            $rules = array(
                'filterName'       => 'required'
            );
            $filters = array(
                'filterName'       => 'trim|sanitize_string'
            );
            $filteredData = $this->valid->filter($_POST, $filters);
            $validate = $this->valid->validate($filteredData, $rules);

            if($validate == true){
                $headers = array(
                    'Content-Type'         => 'application/json',
                    'token'         => $this->session->get('token')
                );

                $data = array(
                    'filterName' => $_POST['filterName'],
                    'filterType' => $_POST['filterType'],
                    'filters' => implode(',', $_POST['filters'])
                );
                $this->log('Data: '. json_encode($data), 'info');
                $result = $this->_model->saveFilter($headers, json_encode($data));
                $this->log('result: '. json_encode($result), 'info');
                switch ($result['status']) {
                    case (200):
                        $response = array(
                            'result' => 'success',
                            'msg' => $this->language->get('filterSaved')
                        );
                        echo json_encode($response);
                        exit;
                    case (421):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('421')
                        );
                        echo json_encode($result);
                        exit();
                    case (500):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('500')
                        );
                        echo json_encode($result);
                        exit();
                    case (0):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('0')
                        );
                        echo json_encode($result);
                        exit();
                    default:
                        $result = array(
                            'result' => 'fail',
                            'code' => $result,
                            'msg' => $this->language->get('default')
                        );
                        echo json_encode($result);
                        exit();
                }
            }else{
                $result = array(
                    'result' => 'fieldFail',
                    'msg' => $this->valid->get_errors_array()
                );
                echo json_encode($result);
            }
        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function polices(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => (Auth::checkCloudPermission('serverBackup') == "true") ? $this->language->get('sPolices') : $this->language->get('bPolices'),
            'active' => 'pms',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/policy/view', $data);
        View::renderTemplate('footer', $data);
    }

    public function getPolicies(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'token' => $this->session->get('token')
            );
            $result = $this->_model->getAllPolicies($headers);
            $this->log('result: '. json_encode($result), 'info');
            switch ($result['status']) {
                case (200):

                    $policies = [];

                    foreach($result['body']->{'backupPolicyElementList'} as $value){
                        $policy = array(
                            'backupFolderPaths' => (array)$value->{'backupFolderPaths'},
                            'cpuUtilization' => (string)$value->{'cpuUtilization'},
                            'memoryUtilization' => (string)$value->{'memoryUtilization'},
                            'crawlLimit' => (string)$value->{'crawlLimit'},
                            'dcmEnabled' => (string)$value->{'dcmEnabled'},
                            'destination' => (string)$value->{'destination'},
                            'filterInclusionName' => (string)$value->{'filterInclusionName'},
                            'lastModifiedTimestamp' => date('d-M-Y h:i:s', (((int)$value->{'lastModifiedTimestamp'}) / 1000)),
                            'lmts' => (string)$value->{'lmts'},
                            'maxFileSize' => (string)$value->{'maxFileSize'},
                            'maxVersions' => (string)$value->{'maxVersions'},
                            'networkThrottlingNames' => (array)$value->{'networkThrottlingNames'},
                            'fileChunkableSize' => (string)$value->{'fileChunkableSize'},
                            'nwFromPGToCloud' => (string)$value->{'nwFromPGToCloud'},
                            'nwFromClientToPG' => (string)$value->{'nwFromClientToPG'},
                            'osType' => (string)$value->{'osType'},
                            'compressionEnabled' => (string)$value->{'compressionEnabled'},
                            'pftEnabled' => (string)$value->{'pftEnabled'},
                            'pftSize' => (string)$value->{'pftSize'},
                            'policyRefresh' => (string)$value->{'policyRefresh'},
                            'dedup' => (string)$value->{'dedup'},
                            'fullBackupEnabled' => (string)$value->{'fullBackupEnabled'},
                            'fullBackupStartDate' => date('d-m-Y', (int)$value->{'fullBackupStartDate'}),
                            'policygroupName' => (string)$value->{'policygroupName'},
                            'privacyGatewayMappingElement' => (array)$value->{'privacyGatewayMappingElement'},
                            'scheduleNames' => (array)$value->{'scheduleNames'},
                            'userSizeAllowed' => (string)$value->{'userSizeAllowed'}
                        );
                        $policies = array_merge(array($policy), $policies);
                    }

                    $response = array(
                        'result' => 'success',
                        'policies' => $policies
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function createPolicy(){
        Auth::allowOnly('admin');
        $schedules = [];
        $filters = [];
        $pgs = [];
        $ccS = [];
        $headers = array(
            'Content-Type'         => 'application/json',
            'token'         => $this->session->get('token')
        );
        $schedulesResult = $this->_model->getSchedules($headers);
        $pgsResult = $this->_model->getPgs($headers);
        $filtersResult = $this->_model->getFilters($headers);
        $ccSResult = $this->_model->getCloudCredentials($headers);

        if($ccSResult['status'] == 200 && !empty($ccSResult['body']->{'cloudCredentialListElement'}->{'cloudCredentialElements'})){
            foreach($ccSResult['body']->{'cloudCredentialListElement'}->{'cloudCredentialElements'} as $key => $value){
                $ccS = array_merge(array($value), $ccS);
            }
        }

        if($schedulesResult['status'] == 200 && $schedulesResult['body']->{'scheduleElementList'}){
            foreach($schedulesResult['body']->{'scheduleElementList'} as $key => $value){
                $schedule = array(
                    'time' => (string)$value->{'time'},
                    'timeZone' => (string)$value->{'timeZone'},
                    'scheduleName' => (string)$value->{'scheduleName'},
                    'daysArray' => (array)$value->{'dayArray'},
                    'days' => $this->getDaysString(implode(',', (array)$value->{'dayArray'})),
                );
                $schedules = array_merge(array($schedule), $schedules);
            }
        }

        if($filtersResult['status'] == 200 && $filtersResult['body']->{'inclusionFilterElementList'}){
            foreach($filtersResult['body']->{'inclusionFilterElementList'} as $key => $value){
                $filter = array(
                    'filters' => (string)$value->{'filters'},
                    'filterName' => (string)$value->{'filterName'},
                    'filterArray' => (array)explode(',', (string)$value->{'filters'})
                );
                $filters = array_merge(array($filter), $filters);
            }
        }

        if($pgsResult['status'] == 200 && $pgsResult['body']->{'privacyGatewayElementList'}){
            foreach($pgsResult['body']->{'privacyGatewayElementList'} as $key => $value){
                $pg = array(
                    'gatewayName' => (string)$value->{'gatewayName'}
                );
                $pgs = array_merge(array($pg), $pgs);
            }
        }

        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => (Auth::checkCloudPermission('serverBackup') == true) ? $this->language->get('sPolicy') : $this->language->get('bPolicy'),
            'crumb2' => $this->language->get('create'),
            'active' => 'pms',
            'schedules' => $schedules,
            'filters' => $filters,
            'pgs' => $pgs,
            'ccS' => $ccS,
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        if(Auth::checkCloudPermission('serverBackup')){
            View::render('pms/serverPolicy/create', $data);
        }else{
            View::render('pms/policy/create', $data);
        }
        View::renderTemplate('footer', $data);
    }

    public function savePolicy(){
        Auth::allowOnly('admin');
        $list = array();
        if(empty($_POST['systemFolders']) || $_POST['systemFolders'] != 'enabled'){
            $_POST['systemFolders'] = 'false';
        }else{
            $_POST['systemFolders'] = 'true';
        }
        if(empty($_POST['legalHold']) || $_POST['legalHold'] != 'enabled'){
            $_POST['legalHold'] = 'false';
        }else{
            $_POST['legalHold'] = 'true';
        }
        if(empty($_POST['maxVersions']) || $_POST['maxVersions'] == 'unlimited'){
            $_POST['maxVersions'] = "-1";
        }
        if(empty($_POST['maxPermissibleFileSize']) || $_POST['maxPermissibleFileSize'] == 'unlimited'){
            $_POST['maxPermissibleFileSize'] = "-1";
        }
        if(empty($_POST['perUserBackupStorageLimit']) || $_POST['perUserBackupStorageLimit'] == 'unlimited'){
            $_POST['perUserBackupStorageLimit'] = "-1";
        }
        if(empty($_POST['machineMaxThreshold']) || $_POST['machineMaxThreshold'] == 'unlimited'){
            $_POST['machineMaxThreshold'] = 100;

        }
        if(!empty($_POST)){
            $rules = array(
                'defaultFolders'            => 'required',
                'filesCrawlLimit'           => 'required',
                'incrementalFileTransfer'   => 'required',
                'machineMaxThreshold'       => 'required',
                'maxPermissibleFileSize'    => 'required',
                'maxVersions'               => 'required',
                'networkThrottling'         => 'required',
                'dcmEnabled'                => 'required',
                'osTyped'                   => 'required',
                'partialFileSize'           => 'required',
                'fileChunkableSize'         => 'required',
                'perUserBackupStorageLimit' => 'required',
                'pgList'                    => 'required',
                'policyName'                => 'required',
                'policyRefresh'             => 'required',
                'schedules'                 => 'required',
                'filters'                   => 'required'
            );
            $filters = array(
                'defaultFolders'            => 'trim|sanitize_string',
                'filesCrawlLimit'           => 'trim|sanitize_string',
                'incrementalFileTransfer'   => 'trim|sanitize_string',
                'machineMaxThreshold'       => 'trim|sanitize_string',
                'maxPermissibleFileSize'    => 'trim|sanitize_string',
                'maxVersions'               => 'trim|sanitize_string',
                'networkThrottling'         => 'trim|sanitize_string',
                'dcmEnabled'                => 'trim|sanitize_string',
                'osTyped'                   => 'trim|sanitize_string',
                'partialFileSize'           => 'trim|sanitize_string',
                'fileChunkableSize'         => 'trim|sanitize_string',
                'perUserBackupStorageLimit' => 'trim|sanitize_string',
                'pgList'                    => 'trim|sanitize_string',
                'policyName'                => 'trim|sanitize_string',
                'policyRefresh'             => 'trim|sanitize_string',
                'filters'                   => 'trim|sanitize_string'
            );
            $filteredData = $this->valid->filter($_POST, $filters);
            $validate = $this->valid->validate($filteredData, $rules);

            if($validate == true){
                $headers = array(
                    'Content-Type'         => 'application/json',
                    'token'         => $this->session->get('token')
                );

                $_POST['defaultFolders'] = explode(',^,',$_POST['defaultFolders']);
                $_POST['pgList'] = explode(',',$_POST['pgList']);

                $privacyGatewayMappingElement = array(
                    'gatewayNwThrottlingName' => 'Default',
                    'privacyGatewayNames' => !empty($_POST['Blukryptipaddress']) ? $_POST['Blukryptipaddress'] : $list
                );

                $data = array(
                    "backupSystemFilesEnabled"      => $_POST['systemFolders'],
                    "legalHoldEnabled"              => $_POST['legalHold'],
                    "shareEnabled"                  => !empty($_POST['shareEnabled']) ? $_POST['shareEnabled'] : 'false',
                    "highSecureShareEnabled"        => !empty($_POST['highSecureShareEnabled']) ? $_POST['highSecureShareEnabled'] : '',
                    "complexPassword"               => ($_POST['highSecureShareEnabled'] == "true") ? true : $_POST['complexPasswordRequired'],
                    "allowDownload"                 => ($_POST['highSecureShareEnabled'] == "true") ? false : $_POST['preventDownloadEnabled'],
                    "shareExpireDateEnable"         => ($_POST['highSecureShareEnabled'] == "true") ? true : $_POST['expiredDate'],
                    "policygroupName"               => !empty($_POST['policyName']) ? $_POST['policyName'] : '',
                    "cloudCredential"               => !empty($_POST['ccS']) ? $_POST['ccS'] : '',
                    "dcmEnabled"                    => !empty($_POST['dcmEnabled']) ? ($_POST['dcmEnabled'] == 'true') ? true : false : false,
                    "destination"                   => !empty($_POST['backupDestination']) ? $_POST['backupDestination'] : '',
                    "backupFolderPaths"             => !empty($_POST['defaultFolders']) ? $_POST['defaultFolders'] : '',
                    "maxVersions"                   => !empty($_POST['maxVersions']) ? $_POST['maxVersions'] : '',
                    "maxFileSize"                   => !empty($_POST['maxPermissibleFileSize']) ? $_POST['maxPermissibleFileSize'] : '',
                    "osType"                        => !empty($_POST['osTyped']) ? $_POST['osTyped'] : '',
                    "pftEnabled"                    => !empty($_POST['incrementalFileTransfer']) ? $_POST['incrementalFileTransfer'] : '',
                    "pftSize"                       => !empty($_POST['partialFileSize']) ? $_POST['partialFileSize'] : '',
                    "dedup"                         => !empty($_POST['dedup']) ? $_POST['dedup'] : 'Disabled',
                    "fullBackupEnabled"             => !empty($_POST['fullBackupEnabled']) ? $_POST['fullBackupEnabled'] : false,
                    "fullBackupStartDate"           => !empty($_POST['fullBackupStartDate']) ? strtotime($_POST['fullBackupStartDate']) : '',
                    "fileChunkableSize"             => !empty($_POST['fileChunkableSize']) ? $_POST['fileChunkableSize'] : 4,
                    "nwFromPGToCloud"               => !empty($_POST['nwFromPGToCloud']) ? $_POST['nwFromPGToCloud'] : 0,
                    "nwFromClientToPG"              => !empty($_POST['nwFromClientToPG']) ? $_POST['nwFromClientToPG'] : 0,
                    "filterInclusionName"           => !empty($_POST['filters']) ? $_POST['filters'] : '',
                    'vssEnabled'                    => !empty($_POST['vssEnabled']) ? $_POST['vssEnabled'] : '',
                    "networkThrottlingNames"        => array('DEFAULT_NETWORK_THROTTLE'),
                    "scheduleNames"                 => !empty($_POST['schedules']) ? $_POST['schedules'] : '',
                    "userSizeAllowed"               => !empty($_POST['perUserBackupStorageLimit']) ? $_POST['perUserBackupStorageLimit'] : '',
                    "memoryUtilization"             => !empty($_POST['memoryUtilization']) ? $_POST['memoryUtilization'] : '',
                    "cpuUtilization"                => !empty($_POST['cpuUtilization']) ? $_POST['cpuUtilization'] : '',
                    "fullBackupSchedule"            => !empty($_POST['fullBackupPeriod']) ? $_POST['fullBackupPeriod'] : '',
                    "compressionEnabled"            => !empty($_POST['compEnabled']) ? $_POST['compEnabled'] : false,
                    "privacyGatewayMappingElement"  => $privacyGatewayMappingElement,
                    "calloutScripts"                => $this->getAllScripts($this->serializeTheString($_POST['preBackupScripts']), $this->serializeTheString($_POST['postBackupScripts']), $this->serializeTheString($_POST['preRestoreScripts']), $this->serializeTheString($_POST['postRestoreScripts'])),
                    "crawlLimit"                    => !empty($_POST['filesCrawlLimit']) ? $_POST['filesCrawlLimit'] : 500,
                    "policyRefresh"                 => !empty($_POST['policyRefresh']) ? $_POST['policyRefresh'] : '',
                    "searchAlgoForPg"                 => !empty($_POST['searchAlgoForPg']) ? $_POST['searchAlgoForPg'] : NEAREST,
                );
                $this->log('Policy Details: '. json_encode($data), 'info');
                $result = $this->_model->savePolicy($headers, json_encode($data));
                switch ($result['status']) {
                    case (200):
                        $response = array(
                            'result' => 'success',
                            'msg' => $this->language->get('policySaved')
                        );
                        echo json_encode($response);
                        exit;
                    case (421):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('421')
                        );
                        echo json_encode($result);
                        exit();
                    case (500):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('500')
                        );
                        echo json_encode($result);
                        exit();
                    case (0):
                        $result = array(
                            'result' => 'fail',
                            'msg' => $this->language->get('0')
                        );
                        echo json_encode($result);
                        exit();
                    default:
                        $result = array(
                            'result' => 'fail',
                            'code' => $result,
                            'msg' => $this->language->get('default')
                        );
                        echo json_encode($result);
                        exit();
                }
            }else{
                $result = array(
                    'result' => 'fieldFail',
                    'msg' => $this->valid->get_errors_array()
                );
                echo json_encode($result);
            }
        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function serializeTheString($oneBigStrings){
        if(!empty($oneBigStrings)){
            $strings = explode('^', $oneBigStrings);
            $array = array();
            foreach($strings as $string){
                list($scriptName, $scriptArgs, $backupType) = explode('|', $string);
                $scriptArray = array(
                    'scriptName' => $scriptName,
                    'backupType' => $backupType,
                    'scriptArgs' => explode(',', $scriptArgs)
                );
                $array = array_merge(array($scriptArray), $array);
            }
            return $array;
        }else{
            return null;
        }
    }

    public function getAllScripts($preBackup, $postBackup, $preRestore, $postRestore){
        $data = [];
        if(!empty($preBackup)){
            foreach($preBackup as $value){
                if($value['backupType'] == 'f'){
                    $preOne = $value['scriptName'].'|preBackupF|'.implode(',',$value['scriptArgs']);
                }else{
                    $preOne = $value['scriptName'].'|preBackupI|'.implode(',',$value['scriptArgs']);
                }
                $data = array_merge(array($preOne), $data);
            }
        }
        if(!empty($postBackup)){
            foreach($postBackup as $value){
                if($value['backupType'] == 'f'){
                    $postOne = $value['scriptName'].'|postBackupF|'.implode(',',$value['scriptArgs']);
                }else{
                    $postOne = $value['scriptName'].'|postBackupI|'.implode(',',$value['scriptArgs']);
                }
                $data = array_merge(array($postOne), $data);
            }
        }
        if(!empty($preRestore)){
            foreach($preRestore as $value){
                if($value['backupType'] == 'f'){
                    $preOne = $value['scriptName'].'|preRestoreF|'.implode(',',$value['scriptArgs']);
                }else{
                    $preOne = $value['scriptName'].'|preRestoreI|'.implode(',',$value['scriptArgs']);
                }
                $data = array_merge(array($preOne), $data);
            }
        }
        if(!empty($postRestore)){
            foreach($postRestore as $value){
                if($value['backupType'] == 'f'){
                    $postOne = $value['scriptName'].'|postRestoreF|'.implode(',',$value['scriptArgs']);
                }else{
                    $postOne = $value['scriptName'].'|postRestoreI|'.implode(',',$value['scriptArgs']);
                }
                $data = array_merge(array($postOne), $data);
            }
        }
        return $data;
    }

    public function editPolicy($policyName){
        Auth::allowOnly('admin');
        $this->session->put('selectedPolicyName', $policyName);
        $schedules = [];
        $filters = [];
        $pgs = [];
        $ccS = [];
        $headers = array(
            'Content-Type'         => 'application/json',
            'token'         => $this->session->get('token')
        );
        $schedulesResult = $this->_model->getSchedules($headers);
        $pgsResult = $this->_model->getPgs($headers);
        $filtersResult = $this->_model->getFilters($headers);
        $ccSResult = $this->_model->getCloudCredentials($headers);

        if($ccSResult['status'] == 200 && !empty($ccSResult['body']->{'cloudCredentialListElement'}->{'cloudCredentialElements'})){
            foreach($ccSResult['body']->{'cloudCredentialListElement'}->{'cloudCredentialElements'} as $key => $value){
                $ccS = array_merge(array($value), $ccS);
            }
        }

        if($schedulesResult['status'] == 200 && $schedulesResult['body']->{'scheduleElementList'}){
            foreach($schedulesResult['body']->{'scheduleElementList'} as $key => $value){
                $schedule = array(
                    'time' => (string)$value->{'time'},
                    'timeZone' => (string)$value->{'timeZone'},
                    'scheduleName' => (string)$value->{'scheduleName'},
                    'daysArray' => (array)$value->{'dayArray'},
                    'days' => $this->getDaysString(implode(',', (array)$value->{'dayArray'})),
                );
                $schedules = array_merge(array($schedule), $schedules);
            }
        }

        if($filtersResult['status'] == 200 && $filtersResult['body']->{'inclusionFilterElementList'}){
            foreach($filtersResult['body']->{'inclusionFilterElementList'} as $key => $value){
                $filter = array(
                    'filters' => (string)$value->{'filters'},
                    'filterName' => (string)$value->{'filterName'},
                    'filterArray' => (array)explode(',', (string)$value->{'filters'})
                );
                $filters = array_merge(array($filter), $filters);
            }
        }

        if($pgsResult['status'] == 200 && $pgsResult['body']->{'privacyGatewayElementList'}){
            foreach($pgsResult['body']->{'privacyGatewayElementList'} as $key => $value){
                $pg = array(
                    'gatewayName' => (string)$value->{'gatewayName'}
                );
                $pgs = array_merge(array($pg), $pgs);
            }
        }

        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => (Auth::checkCloudPermission('serverBackup') == true) ? $this->language->get('sPolicy') : $this->language->get('bPolicy'),
            'crumb2' => $this->language->get('edit'),
            'active' => 'pms',
            'schedules' => $schedules,
            'filters' => $filters,
            'pgs' => $pgs,
            'ccS' => $ccS,
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        if(Auth::checkCloudPermission('serverBackup')){
            View::render('pms/serverPolicy/edit', $data);
        }else{
            View::render('pms/policy/edit', $data);
        }
        View::renderTemplate('footer', $data);
    }

    public function getPolicy(){
        Auth::allowOnly('admin');
        $policyName = !empty($this->session->get('selectedPolicyName')) ? $this->session->get('selectedPolicyName') : $this->session->get('policyName');
        if(!empty($_POST) && !empty($policyName)){
            $headers = array(
                'token' => $this->session->get('token')
            );
            $result = $this->_model->getPolicy($headers, $policyName);
            $this->log('result'.json_encode($result), 'info');
            switch ($result['status']) {
                case (200):

                    $value = $result['body']->{'backupPolicyElement'};
                    $policy = array(
                        'backupSystemFilesEnabled' => $value->{'backupSystemFilesEnabled'},
                        'legalHoldEnabled' => $value->{'legalHoldEnabled'},
                        'backupFolderPaths' => (array)$value->{'backupFolderPaths'},
                        'cpuUtilization' => (string)$value->{'cpuUtilization'},
                        'memoryUtilization' => (string)$value->{'memoryUtilization'},
                        'nwFromPGToCloud' => (string)$value->{'nwFromPGToCloud'},
                        'nwFromClientToPG' => (string)$value->{'nwFromClientToPG'},
                        'crawlLimit' => (string)$value->{'crawlLimit'},
                        'dcmEnabled' => (string)$value->{'dcmEnabled'},
                        'destination' => (string)$value->{'destination'},
                        'filterInclusionName' => (string)$value->{'filterInclusionName'},
                        'lastModifiedTimestamp' => date('d-M-Y h:i:s', (((int)$value->{'lastModifiedTimestamp'}) / 1000)),
                        'lmts' => (string)$value->{'lmts'},
                        'maxFileSize' => (string)$value->{'maxFileSize'},
                        'maxVersions' => (string)$value->{'maxVersions'},
                        'shareEnabled' => (boolean)$value->{'shareEnabled'},
                        'complexPassword' => (boolean)$value->{'complexPassword'},
                        'shareExpireDateEnable' => (boolean)$value->{'shareExpireDateEnable'},
                        'allowDownload' => (boolean)$value->{'allowDownload'},
                        'highSecureShareEnabled' => (boolean)$value->{'highSecureShareEnabled'},
                        'networkThrottlingNames' => (array)$value->{'networkThrottlingNames'},
                        'osType' => (string)$value->{'osType'},
                        'calloutScripts' => $value->{'calloutScripts'},
                        'compressionEnabled' => $value->{'compressionEnabled'},
                        'pftEnabled' => (string)$value->{'pftEnabled'},
                        'pftSize' => (string)$value->{'pftSize'},
                        'vssEnabled' => (string)$value->{'vssEnabled'},
                        'fileChunkableSize' => (string)$value->{'fileChunkableSize'},
                        'fullBackupSchedule' => (string)$value->{'fullBackupSchedule'},
                        'policyRefresh' => (string)$value->{'policyRefresh'},
                        'dedup' => (string)$value->{'dedup'},
                        'fullBackupEnabled' => (string)$value->{'fullBackupEnabled'},
                        'fullBackupStartDate' => date('d-m-Y', (int)$value->{'fullBackupStartDate'}),
                        'policygroupName' => (string)$value->{'policygroupName'},
                        'privacyGatewayMappingElement' => (array)$value->{'privacyGatewayMappingElement'},
                        'scheduleNames' => (array)$value->{'scheduleNames'},
                        'userSizeAllowed' => (string)$value->{'userSizeAllowed'}
                    );

                    $response = array(
                        'result' => 'success',
                        'policy' => $policy
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function getUserPolicy(){
        //Auth::allowOnly('admin');
        $policyName = $this->session->get('policyName');
        if(!empty($_POST) && !empty($policyName)){
            $headers = array(
                'token' => $this->session->get('token')
            );
            $result = $this->_model->getPolicy($headers, $policyName);
            $this->log('result'.json_encode($result), 'info');
            switch ($result['status']) {
                case (200):

                    $value = $result['body']->{'backupPolicyElement'};
                    $policy = array(
                        'backupSystemFilesEnabled' => $value->{'backupSystemFilesEnabled'},
                        'legalHoldEnabled' => $value->{'legalHoldEnabled'},
                        'backupFolderPaths' => (array)$value->{'backupFolderPaths'},
                        'cpuUtilization' => (string)$value->{'cpuUtilization'},
                        'memoryUtilization' => (string)$value->{'memoryUtilization'},
                        'nwFromPGToCloud' => (string)$value->{'nwFromPGToCloud'},
                        'nwFromClientToPG' => (string)$value->{'nwFromClientToPG'},
                        'crawlLimit' => (string)$value->{'crawlLimit'},
                        'dcmEnabled' => (string)$value->{'dcmEnabled'},
                        'destination' => (string)$value->{'destination'},
                        'filterInclusionName' => (string)$value->{'filterInclusionName'},
                        'lastModifiedTimestamp' => date('d-M-Y h:i:s', (((int)$value->{'lastModifiedTimestamp'}) / 1000)),
                        'lmts' => (string)$value->{'lmts'},
                        'maxFileSize' => (string)$value->{'maxFileSize'},
                        'maxVersions' => (string)$value->{'maxVersions'},
                        'shareEnabled' => (boolean)$value->{'shareEnabled'},
                        'complexPassword' => (boolean)$value->{'complexPassword'},
                        'shareExpireDateEnable' => (boolean)$value->{'shareExpireDateEnable'},
                        'allowDownload' => (boolean)$value->{'allowDownload'},
                        'highSecureShareEnabled' => (boolean)$value->{'highSecureShareEnabled'},
                        'networkThrottlingNames' => (array)$value->{'networkThrottlingNames'},
                        'osType' => (string)$value->{'osType'},
                        'calloutScripts' => $value->{'calloutScripts'},
                        'compressionEnabled' => $value->{'compressionEnabled'},
                        'pftEnabled' => (string)$value->{'pftEnabled'},
                        'pftSize' => (string)$value->{'pftSize'},
                        'vssEnabled' => (string)$value->{'vssEnabled'},
                        'fileChunkableSize' => (string)$value->{'fileChunkableSize'},
                        'fullBackupSchedule' => (string)$value->{'fullBackupSchedule'},
                        'policyRefresh' => (string)$value->{'policyRefresh'},
                        'dedup' => (string)$value->{'dedup'},
                        'fullBackupEnabled' => (string)$value->{'fullBackupEnabled'},
                        'fullBackupStartDate' => date('d-m-Y', (int)$value->{'fullBackupStartDate'}),
                        'policygroupName' => (string)$value->{'policygroupName'},
                        'privacyGatewayMappingElement' => (array)$value->{'privacyGatewayMappingElement'},
                        'scheduleNames' => (array)$value->{'scheduleNames'},
                        'userSizeAllowed' => (string)$value->{'userSizeAllowed'}
                    );

                    $response = array(
                        'result' => 'success',
                        'policy' => $policy
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function singlePolicyMapping(){
        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('policy_mapping'),
            'crumb2' => $this->language->get('change_user_policy'),
            'active' => 'pms',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/policyMapping/single', $data);
        View::renderTemplate('footer', $data);
    }

    public function getPolicyMappingDetails(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $_POST['skipValue'] = ($_POST['skipValue'] != 0 ) ? $_POST['skipValue']+1 : $_POST['skipValue'];
            $headers = array(
                'token' => $this->session->get('token'),
                'skipValue' => !empty($_POST['skipValue']) ? (int)$_POST['skipValue'] : 0
            );
            if(!empty($_POST['userName']) && $_POST['userName'] != 'all'){
                $headers['userNames'] = $_POST['userName'];
                $result = $this->_model->searchPolicyMapping($headers);
            }else{
                $result = $this->_model->getPolicyMappingDetails($headers);
            }
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'policies' => $result['body']->{'editUserBackupPolicyMappingElement'}->{'existingPolicies'},
                        'users' => $result['body']->{'editUserBackupPolicyMappingElement'}->{'userBackupPolicyMappingElements'}
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $response = array(
                        'result' => 'fail',
                        'response' => $result,
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($response);
                    exit();
                case (500):
                    $response = array(
                        'result' => 'fail',
                        'response' => $result,
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($response);
                    exit();
                case (0):
                    $response = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($response);
                    exit();
                default:
                    $response = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($response);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function saveSinglePolicyMapping(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'Content-Type'      => 'application/json',
                'loginUserName'     => $this->session->get('userName'),
                'userName'          => $_POST['userName'],
                'newPolicyName'     => $_POST['newPolicyName'],
                'existingPolicyName'=> $_POST['oldPolicyName']
            );
            $result = $this->_model->saveSinglePolicyMapping($headers, $_POST['userName']);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'response' => $result,
                        'msg'    => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function multiplePolicyMapping(){
        Auth::allowOnly('admin');
        $result = $this->_model->getAllPolicies(array('token'=>$this->session->get('token')));

        if(!empty((array)$result['body']->{'backupPolicyElementList'})){
            $policies = [];
            foreach((array)$result['body']->{'backupPolicyElementList'} as $value){
                $policy = array('name' => $value->{'policygroupName'});
                $policies = array_merge(array($policy), $policies);
            }
            $data = array(
                'title' => $this->language->get('indexPageName'),
                'crumb1' => $this->language->get('policy_mapping'),
                'crumb2' => (Auth::checkCloudPermission('serverBackup')) ? $this->language->get('assign_policy') : $this->language->get('assign_policy_for_users_in_bulk'),
                'active' => 'pms',
                'csrf_token' => $this->setCSRFToken(),
                'policies' => $policies,
                'loggedInUser' => $this->session->all()
            );

            View::renderTemplate('header', $data);
            View::renderTemplate('navigation', $data);
            View::render('pms/policyMapping/multiple', $data);
            View::renderTemplate('footer', $data);
        }else{
            header('Location: '.Url::basePath('pms/policy/create'));
        }
    }

    public function saveMultiplePolicyMapping(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'Content-Type'      => 'application/json',
                'loginUserName'     => $this->session->get('userName'),
                'userNames'          => $_POST['userNames'],
                'policyName'     => $_POST['policyName']
            );
            $result = $this->_model->saveMultiplePolicyMapping($headers);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'response' => $result,
                        'msg'    => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function paracloudblukrypt(){
        Auth::allowOnly('admin');

        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('paracloudblukrypt'),
            'active' => 'settings',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/machine/paracloudblukrypt', $data);
        View::renderTemplate('footer', $data);
    }

    public function getBluvault(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'Content-Type' =>  'application/json',
                'token' =>  $this->session->get('token'),
            );
            $result = $this->_model->getBluvaults($headers);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'bluvaults' => $result['body']->{'bluVaultMappingElementList'}
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function saveBluvault(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            if($this->checkForBluVaultInstallation($_POST['bluvault']) == false){
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('bluVaultNotInstalled')
                );
                echo json_encode($result);
                exit();
            }
            $headers = array(
                'Content-Type'      => 'application/json',
                'token' =>  $this->session->get('token')
            );
            $result = $this->_model->saveBluvault($headers, json_encode(array('bluVaultName' => $_POST['bluvault'])));
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'response' => $result,
                        'msg'    => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function deleteBluvault(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'Content-Type'      => 'application/json',
                'token' =>  $this->session->get('token'),
                'bluvault-id'      => $_POST['bluvaultId']
            );
            $result = $this->_model->deleteBluvault($headers);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'response' => $result,
                        'msg'    => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function blukrypt(){
        Auth::allowOnly('admin');

        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('blukrypt'),
            'active' => 'settings',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/machine/blukrypt', $data);
        View::renderTemplate('footer', $data);
    }

    public function getAllBlukrypts(){
        if(!empty($_POST)){

            $headers = array(
                'Content-Type'      => 'application/json',
                'token' =>  $this->session->get('token')
            );
            $result = $this->_model->getAllBlukrypts($headers);

            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result'    => 'success',
                        'blukrypts' => $result['body']->{'privacyGatewayElementList'},
                        'msg'       => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function saveBlukrypt(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            if($this->checkForBluVaultInstallation($_POST['blukrypt']) == false){
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('bluKryptNotInstalled')
                );
                echo json_encode($result);
                exit();
            }
            $headers = array(
                'Content-Type'      => 'application/json',
                'token'             =>  $this->session->get('token'),
                'blukrypt'          => $_POST['blukrypt'],
                'blukryptType'      => $_POST['blukryptType']
            );
            $result = $this->_model->saveBlukrypt($headers);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'response' => $result,
                        'msg'    => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function removeBlukrypt(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'Content-Type'      => 'application/json',
                'token' =>  $this->session->get('token'),
                'blukrypt'      => $_POST['bluKrypt']
            );
            $result = $this->_model->deleteBluKrypt($headers);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'response' => $result,
                        'msg'    => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function networkThrottling(){
        Auth::allowOnly('admin');
        $headers = array(
            'Content-Type' => 'application/json',
            'token' => $this->session->get('token')
        );
        $result = $this->_model->getPgLists($headers);
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('networkThrottling'),
            'listNetworkThrottling' => $this->language->get('listNetworkThrottling'),
            'active' => 'pms',
            'pgList' => $result['body']->{'privacyGatewayElementList'},
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/networkThrottling/view', $data);
        View::renderTemplate('footer', $data);
    }

    public function networkThrottlingSet($type, $pg){
        Auth::allowOnly('admin');
        $headers = array(
            'Content-Type' => 'application/json',
            'token' => $this->session->get('token')
        );
        $result = $this->_model->getPgLists($headers);
        if($this->checkIfExists($pg, $result['body']->{'privacyGatewayElementList'})){
            $exists = true;
        }else{
            $exists = false;
        }
        $nTResult = $this->_model->getNTList($headers, $pg);
        $c2p = [];
        $p2a = [];
        if(!empty($nTResult['body']->{'networkThrottlingElementList'})){
            foreach($nTResult['body']->{'networkThrottlingElementList'} as $value){
                $newItem = array(
                    'id' => $value->{'id'},
                    'networkThrottlingName' => $value->{'networkThrottlingName'},
                    'privacyGatewayName' => $value->{'privacyGatewayName'},
                    'startTime' => $value->{'startTime'},
                    'endTime' => $value->{'endTime'},
                    'uploadSpeed' => $value->{'uploadSpeed'},
                    'downloadSpeed' => $value->{'downloadSpeed'},
                    'throttleType' => $value->{'throttleType'},
                );
                if($value->{'throttleType'} == 'c2p'){
                    $c2p = array_merge(array($newItem), $c2p);
                }elseif($value->{'throttleType'} == 'p2a'){
                    $p2a = array_merge(array($newItem), $p2a);
                }
            }
        }

        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('networkThrottling'),
            'listNetworkThrottling' => $this->language->get('listNetworkThrottling'),
            'active' => 'pms',
            'pgList' => $result['body']->{'privacyGatewayElementList'},
            'selectedPG' => $pg,
            'type' => $type,
            'c2p' => $c2p,
            'p2a' => $p2a,
            'exists' => $exists,
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/networkThrottling/set', $data);
        View::renderTemplate('footer', $data);
    }

    public function saveThrottlingLimit(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'Content-Type'              => 'application/json',
                'token'                     => $this->session->get('token')
            );
            $data = array(
                'privacyGatewayName'        => $_POST['pgName'],
                'throttleType'              => $_POST['type'],
                'startTime'                 => $_POST['startTime'],
                'endTime'                   => $_POST['endTime'],
                'uploadSpeed'               => $_POST['uploadSpeed']
            );
            $result = $this->_model->saveThrottlingLimit($headers, json_encode($data));
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'response' => $result,
                        'msg'    => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'msg' => $this->language->get('noData')
                )
            );
        }
    }

    public function removeThrottlingLimit(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'Content-Type'              => 'application/json',
                'token'                     => $this->session->get('token'),
                'throttle-id'                     => $_POST['id']
            );
            $result = $this->_model->removeThrottlingLimit($headers);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'response' => $result,
                        'msg'    => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'msg' => $this->language->get('noData')
                )
            );
        }
    }

    public function serverSinglePolicyMapping(){
        Auth::allowOnly('admin');

        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('Server'),
            'crumb2' => $this->language->get('Single Policy Mapping'),
            'active' => 'pms',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/serverPolicyMapping/single', $data);
        View::renderTemplate('footer', $data);
    }

    public function serverMultiplePolicyMapping(){
        Auth::allowOnly('admin');

        $result = $this->_model->getAllPolicies(array('token'=>$this->session->get('token')));

        if(!empty((array)$result['body']->{'backupPolicyElementList'})){
            $policies = [];
            foreach((array)$result['body']->{'backupPolicyElementList'} as $value){
                $policy = array('name' => $value->{'policygroupName'});
                $policies = array_merge(array($policy), $policies);
            }

            $data = array(
                'title' => $this->language->get('indexPageName'),
                'crumb1' => $this->language->get('Server'),
                'crumb2' => $this->language->get('Multiple Policy Mapping'),
                'active' => 'pms',
                'csrf_token' => $this->setCSRFToken(),
                'loggedInUser' => $this->session->all(),
                'policies' => $policies,
            );

            View::renderTemplate('header', $data);
            View::renderTemplate('navigation', $data);
            View::render('pms/serverPolicyMapping/multiple', $data);
            View::renderTemplate('footer', $data);
        }else{
            header('Location: /pms/policy/create');
        }
    }

    public function getServerPolicyMappingList(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $_POST['skipValue'] = ($_POST['skipValue'] != 0 ) ? $_POST['skipValue']+1 : $_POST['skipValue'];
            $headers = array(
                'token' => $this->session->get('token'),
                'skipValue' => !empty($_POST['skipValue']) ? (int)$_POST['skipValue'] : 0
            );
            if(!empty($_POST['userName']) && $_POST['userName'] != 'all'){
                $headers['deviceNames'] = $_POST['userName'];
                $result = $this->_model->searchPolicyMapping($headers);
            }else{
                $result = $this->_model->getServerPolicyMappingList($headers);
            }
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'policies' => $result['body']->{'editDeviceBackupPolicyMappingElement'}->{'existingPolicies'},
                        'users' => $result['body']->{'editDeviceBackupPolicyMappingElement'}->{'deviceBackupPolicyMappingElements'}
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $response = array(
                        'result' => 'fail',
                        'response' => $result,
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($response);
                    exit();
                case (500):
                    $response = array(
                        'result' => 'fail',
                        'response' => $result,
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($response);
                    exit();
                case (0):
                    $response = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($response);
                    exit();
                default:
                    $response = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($response);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
        }
    }

    public function saveServerPolicyMapping(){
        Auth::allowOnly('admin');
        if(!empty($_POST)){
            $headers = array(
                'Content-Type'      => 'application/json',
                'loginUserName'     => $this->session->get('userName'),
                'devices'           => $_POST['deviceUUID'],
                'policyName'        => $_POST['newPolicyName']
            );
            $this->log('Headers'. json_encode($headers), 'info');
            $result = $this->_model->saveServerPolicyMapping($headers);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'response' => $result,
                        'msg'    => 'done'
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    /**
     * Below Method Renders Encryption settings page
     */
    public function cloudCredentials(){

        Auth::allowOnly('admin');
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'crumb1' => $this->language->get('cloudCredentials'),
            'active' => 'settings',
            'csrf_token' => $this->setCSRFToken(),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('pms/ccS/cloudCredentials', $data);
        View::renderTemplate('footer', $data);
    }

    public function getAllCcS(){
        Auth::allowOnly('admin');
        if(empty($_POST)){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
        $ccS = [];
        $headers = array(
            'Content-Type'      => 'application/json',
            'token'          => $this->session->get('token')
        );
        $result = $this->_model->getCloudCredentials($headers);
        switch ($result['status']) {
            case (200):
                if(!empty($result['body']->{'cloudCredentialListElement'}->{'cloudCredentialElements'})){
                    foreach($result['body']->{'cloudCredentialListElement'}->{'cloudCredentialElements'} as $value){
                        switch($value->cloudName){
                            case 'azureBlob':
                                $value->cloudNameInCaps = 'Azure Blob';
                                break;
                            case 'ibmCloud':
                                $value->cloudNameInCaps = 'IBM Cloud';
                                break;
                            case 'amazon':
                                $value->cloudNameInCaps = 'Amazon s3 Services';
                                break;
                            default:
                                $value->cloudNameInCaps = '';
                                break;
                        }

                        $ccS = array_merge(array($value), $ccS);
                    }
                }
                $response = array(
                    'result' => 'success',
                    'ccS' => $ccS,
                );
                echo json_encode($response);
                exit;

            case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
            case (0):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }

    public function saveCloudCredentials(){
        Auth::allowOnly('admin');
        if(!empty($_POST['cloudType'])){
            $data = array(
                'cloudName' => $_POST['cloudType'],
                'identity' => ($_POST['cloudType'] == "localStorage") ? "local" : $_POST['identity'],
                'credential' => $_POST['accessKey'],
                'containerName' => !empty($_POST['containerName']) ? $_POST['containerName'] : '',
                'endPointUrl' => ($_POST['cloudType'] == "localStorage") ? $_POST['localStorage'] : !empty($_POST['epaUrl']) ? $_POST['epaUrl'] : '',
            );
            $headers = array(
                'Content-Type'      => 'application/json',
                'token'          => $this->session->get('token')
            );
            $result = $this->_model->saveCloudCredentials($headers, $data);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'msg'    => $this->language->get('savedCloudCredentials')
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function removeCloudCredentials(){
        Auth::allowOnly('admin');
        if(!empty($_POST['identity'])){
            $headers = array(
                'Content-Type'      => 'application/json',
                'identity'          => $_POST['identity'],
                'token'             => $this->session->get('token')
            );
            $result = $this->_model->removeCloudCredentials($headers);
            $this->log("headers: ". json_encode($headers), 'info');
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'msg'    => $this->language->get('removeCloudCredentials')
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'msg' => $this->language->get('noData')
                )
            );
        }
    }

    public function checkForBluVaultInstallation($bluvault){
        if (strpos($bluvault,'localhost') !== false) {
            return true;
        }else{
            $result = $this->_model->checkForBluVaultInstallation($bluvault);
            return ($result['status'] != 500 && $result['status'] != 0);
        }
    }

    public function checkForBluKryptInstallation($bluKrypt){
        if (strpos($bluKrypt,'localhost') !== false) {
            return true;
        }else{
            $result = $this->_model->checkForBluKryptInstallation($bluKrypt);
            return ($result['status'] != 500 && $result['status'] != 0);
        }
    }

}
