<?php

namespace Controllers;

use Core\Language;
use Core\View;
use Core\Controller;
use Helpers\Url;
use Models\Backup;
use ZMQ;
use ZMQContext;
use ZMQPoll;
use ZMQSocket;
/*
 * Copyright 2014
 * ParaBlu Systems Private Limited
 * All Rights Reserved
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of ParaBlu Systems Private Limited and its suppliers, if any.
 * The intellectual and technical concepts contained herein are proprietary
 * to ParaBlu Systems Private Limited and its suppliers and may be covered by
 * Indian, US and Foreign Patents, patents in process, and are protected by
 * trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from ParaBlu Systems Private Limited.
 *
 * Login controller
 *
 * @author Prashanth Kumar B - prashanth.kumar@parablu.com
 * @version 2.0
 * @date June 10, 2015
 *
 */
class BackupController extends Controller{

    /**
     * Below variable contains the object reference of a model
     */

    protected $_model;

    /**
     * Call the parent construct
     */
    public function __construct(){
        parent::__construct();
        $this->auth->loginCheck();
        $this->language->load('backup');
        $this->_model = new Backup();
        $this->verifyAndCleanItUp();
    }

    public function devices(){
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'csrf_token' => $this->setCSRFToken(),
            'crumb1' => $this->language->get('devices'),
            'active' => 'backup',
            'selectedUser' => $this->session->get('userName'),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('backup/devices', $data);
        View::renderTemplate('footer', $data);
    }

    public function restoreFiles($deviceId, $deviceName, $deviceUUID){
        $this->session->put('currentDeviceUUID', $deviceUUID);
        $this->session->put('currentDeviceId', $deviceId);
        $this->session->put('currentDeviceName', $deviceName);
        if(empty($this->session->get('previousDeviceId'))){
            $this->session->put('previousDeviceId', $deviceId);
        }elseif($this->session->get('previousDeviceId') != $deviceId){
            $this->session->put('previousDeviceId', $deviceId);
            $this->session->put('filePath', '');
        }
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'csrf_token' => $this->setCSRFToken(),
            'crumb1' => $this->language->get('device'),
            'crumb2' => $deviceName,
            'active' => 'backup',
            'crumb3' => $this->language->get('files'),
            'loggedInUser' => $this->session->all()
        );

        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('backup/restoreFiles', $data);
        View::renderTemplate('footer', $data);
    }

    public function files($deviceId, $deviceName, $deviceUUID){
        $this->session->put('currentDeviceUUID', $deviceUUID);
        $this->session->put('currentDeviceId', $deviceId);
        $this->session->put('currentDeviceName', $deviceName);
        if(empty($this->session->get('previousDeviceId'))){
            $this->session->put('previousDeviceId', $deviceId);
        }elseif($this->session->get('previousDeviceId') != $deviceId){
            $this->session->put('previousDeviceId', $deviceId);
            $this->session->put('filePath', '');
        }
        $data = array(
            'title' => $this->language->get('indexPageName'),
            'csrf_token' => $this->setCSRFToken(),
            'crumb1' => $this->language->get('device'),
            'crumb2' => $deviceName,
            'active' => 'backup',
            'policy' => $this->getPolicy(),
            'crumb3' => $this->language->get('files'),
            'loggedInUser' => $this->session->all()
        );
        View::renderTemplate('header', $data);
        View::renderTemplate('navigation', $data);
        View::render('backup/files', $data);
        View::renderTemplate('footer', $data);
    }

    public function getPolicy(){
        $policyName = $this->session->get('policyName');
        if(!empty($policyName)){
            $headers = array(
                'token' => $this->session->get('token')
            );
            $result = $this->_model->getPolicy($headers, $policyName);
            $this->log('result'.json_encode($result), 'info');
            if($result['status'] == 200){
                $value = $result['body']->{'backupPolicyElement'};
                return array(
                    'backupSystemFilesEnabled' => $value->{'backupSystemFilesEnabled'},
                    'legalHoldEnabled' => $value->{'legalHoldEnabled'},
                    'backupFolderPaths' => (array)$value->{'backupFolderPaths'},
                    'cpuUtilization' => (string)$value->{'cpuUtilization'},
                    'memoryUtilization' => (string)$value->{'memoryUtilization'},
                    'nwFromPGToCloud' => (string)$value->{'nwFromPGToCloud'},
                    'nwFromClientToPG' => (string)$value->{'nwFromClientToPG'},
                    'crawlLimit' => (string)$value->{'crawlLimit'},
                    'dcmEnabled' => (string)$value->{'dcmEnabled'},
                    'destination' => (string)$value->{'destination'},
                    'filterInclusionName' => (string)$value->{'filterInclusionName'},
                    'lastModifiedTimestamp' => date('d-M-Y h:i:s', (((int)$value->{'lastModifiedTimestamp'}) / 1000)),
                    'lmts' => (string)$value->{'lmts'},
                    'maxFileSize' => (string)$value->{'maxFileSize'},
                    'maxVersions' => (string)$value->{'maxVersions'},
                    'shareEnabled' => (string)$value->{'shareEnabled'},
                    'complexPassword' => (string)$value->{'complexPassword'},
                    'shareExpireDateEnable' => (string)$value->{'shareExpireDateEnable'},
                    'allowDownload' => (string)$value->{'allowDownload'},
                    'highSecureShareEnabled' => (string)$value->{'highSecureShareEnabled'},
                    'networkThrottlingNames' => (array)$value->{'networkThrottlingNames'},
                    'osType' => (string)$value->{'osType'},
                    'calloutScripts' => $value->{'calloutScripts'},
                    'compressionEnabled' => $value->{'compressionEnabled'},
                    'pftEnabled' => (string)$value->{'pftEnabled'},
                    'pftSize' => (string)$value->{'pftSize'},
                    'vssEnabled' => (string)$value->{'vssEnabled'},
                    'fileChunkableSize' => (string)$value->{'fileChunkableSize'},
                    'fullBackupSchedule' => (string)$value->{'fullBackupSchedule'},
                    'policyRefresh' => (string)$value->{'policyRefresh'},
                    'policygroupName' => (string)$value->{'policygroupName'},
                    'privacyGatewayMappingElement' => (array)$value->{'privacyGatewayMappingElement'},
                    'scheduleNames' => (array)$value->{'scheduleNames'},
                    'userSizeAllowed' => (string)$value->{'userSizeAllowed'}
                );
            }else{
                return array();
            }

        }else{
            return array();
        }
    }
    public function isShareCheck(){

        if(!empty($_POST)){
            $stat = $this->collectStats();
            $headers = array(
                'Content-Type'      => 'application/json',
                'token'             => $this->session->get('token'),
                'productType'       => "backup",
                'backupId' => $_POST['backupID']
            );
            if($_POST['type']){
                $headers['osType']  = $this->getOS($_SERVER['HTTP_USER_AGENT']);
                $headers['browser'] = $_SERVER['SERVER_NAME'];
                $headers['ip']      = $stat['remoteAdd'];
                $headers['accessType']    = $_POST['type'];
            }

            $headers['miniCloudName'] = '';
            $headers['devicePath'] = htmlspecialchars_decode($_POST['filePath']);
            $headers['fileName'] = $_POST['fileName'];

            $result = $this->_model->isSharedOrNot($headers);
            if($result['status'] == 302){
                $shared = $result['body']->{'sharedFileImageElement'};
                if($shared->{'passwordProtected'} == true){
                    $type = md5('protected');
                }else{
                    $type = md5('read');
                }
                $link = SHARE_PATH.$type.DS.$shared->{'key12'}.DS.$shared->{'key14'}.DS.$shared->{'key18'}.DS.$shared->{'key20'};
                $isShared = true;
                $shared->{'validUntil'} = !(empty($shared->{'validUntil'})) ? date('m/d/Y', strtotime(date('m/d/y', (((int)$shared->{'validUntil'}) / 1000)) . ' -1 day')) : null;
            }else{
                $shared = array();
                $link = '';
                $isShared = false;
            }

            $response = array(
                'data' => $shared,
                'result' => $isShared,
                'link' => $link
            );
            echo json_encode($response);
            exit;

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function getFiles(){
        if(empty($_POST) && empty($_POST['filePath'])){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
            exit;
        }

        if($_POST['refresh'] == 'true'){
            $filePath = (!empty($this->session->get('filePath'))) ? $this->session->get('filePath') : '';
        }else{
            $filePath = ($_POST['filePath'] == 'root') ? '' : $_POST['filePath'];
        }
        $headers = array(
            'filePath'      => html_entity_decode($filePath, ENT_QUOTES | ENT_HTML5),
            'deviceId'      => $this->session->get('currentDeviceId'),
            'token'         => $this->session->get('token')
        );
        $result = $this->_model->getFiles($headers);
        $this->log('result: '.json_encode($result), 'info');
        switch ($result['status']) {
            case (200):
                $items = array();
                if(!empty($result['body']->{'backup-file'})){
                    foreach($result['body']->{'backup-file'} as $value){
                        if($value->{'folder'} == 'true'){
                            $folder = array(
                                'backupId' => (string)$value->{'backupId'},
                                'checksum' => (string)$value->{'checksum'},
                                'exists' => (string)$value->{'exists'},
                                'name' => (string)$value->{'file-name'},
                                'user' => (string)$value->{'file-user'},
                                'type' => 'Folder',
                                'icon' => 'folder',
                                'size' => '',
                                'filePath' => (string)$value->{'file-path'},
                                'fullPath' => (!empty((string)$value->{'file-path'})) ? (string)$value->{'file-path'}.'/'.(string)$value->{'file-name'} : (string)$value->{'file-name'},
                                'fsPath' => (string)$value->{'fs-path'},
                                'baseFolder' => (string)$value->{'is-basefolder'},
                                'externalStorage' => (string)$value->{'is-externalStorage'},
                                'clientModifiedTime' => (!empty($value->{'client-modified-time'}) && $value->{'client-modified-time'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'client-modified-time'}) / 1000)) : '',
                                'serverModifiedTime' => (!empty($value->{'server-modified-time'}) && $value->{'server-modified-time'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'server-modified-time'}) / 1000)) : '',
                                'status' => (string)$value->{'status'}
                            );
                            $items = array_merge(array($folder), $items);
                        }else{
                            $fileExt = explode('.', (string)$value->{'file-name'});
                            $file = array(
                                'backupId' => (string)$value->{'backupId'},
                                'checksum' => (string)$value->{'checksum'},
                                'exists' => (string)$value->{'exists'},
                                'name' => (string)$value->{'file-name'},
                                'user' => (string)$value->{'file-user'},
                                'type' => 'File',
                                'filePath' => (string)$value->{'file-path'},
                                'fullPath' => (!empty((string)$value->{'file-path'})) ? (string)$value->{'file-path'}.'/'.(string)$value->{'file-name'} : (string)$value->{'file-name'},
                                'icon' => $this->getFileIcon(end($fileExt)),
                                'fsPath' => (string)$value->{'fs-path'},
                                'baseFolder' => (string)$value->{'is-basefolder'},
                                'storagePlace' => (!empty((string)$value->{'storagePlace'})) ? (string)$value->{'storagePlace'} : 'PG',
                                'externalStorage' => (string)$value->{'is-externalStorage'},
                                'clientModifiedTime' => (!empty($value->{'client-modified-time'}) && $value->{'client-modified-time'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'client-modified-time'}) / 1000)) : '',
                                'serverModifiedTime' => (!empty($value->{'server-modified-time'}) && $value->{'server-modified-time'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'server-modified-time'}) / 1000)) : '',
                                'rawSize' => (int)(string)$value->{'size'},
                                'size' => $this->getFileSize((string)$value->{'size'}),
                                'status' => (string)$value->{'status'}
                            );
                            $items = array_merge(array($file), $items);
                        }
                    }
                }
                if(!empty($items)){
                    $this->session->put('filePath', $filePath);
                }
                $response = array(
                    'result' => 'success',
                    'post' => html_entity_decode($filePath, ENT_QUOTES | ENT_HTML5),
                    'items' => $items
                );
                if($_POST['refresh'] == 'true' && !empty($filePath)){
                    $response['crumb'] = explode('/', $filePath);
                    $response['crumbString'] = $filePath;
                }
                echo json_encode($response);
                exit;
            case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
            case (0):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }

    public function getDevices(){
        if(!empty($_POST) && !empty($_POST['userName'])){
            $headers = array(
                'token'         => $this->session->get('token')
            );
            $result = $this->_model->getDevices($headers, !empty($_POST['userName']) ? $_POST['userName'] : $this->session->get('userName'));
            switch ($result['status']) {
                case (200):
                    $devices = array();
                    if(!empty($result['body']->{'device-info'})){
                        foreach($result['body']->{'device-info'} as $value){
                            $device = array(
                                'name'          => (string)$value->{'device-name'},
                                'userName'      => (string)$value->{'userName'},
                                'decoupled'     => (string)$value->{'decoupled'},
                                'id'            => (string)$value->{'device-id'},
                                'uuId'          => (string)$value->{'device-uuid'},
                                'tempPath'      => (string)$value->{'device-temp-path'},
                                'status'        => ($value->{'is-blocked'} == 'false') ? 'Active' : 'In-Active',
                                'blocked'       => (string)$value->{'is-blocked'},
                                'os'            => $this->getOsType($value->{'device-os-type'}),
                                'version'       => (string)$value->{'device-os-version'},
                                'icon'          => $this->getOsIcon($value->{'device-os-type'}),
                                'modifiedOn'    => date('d-M-Y h:i:s', (((int)$value->{'modified-timestamp'}) / 1000)),
                                'createdOn'     => date('d-M-Y h:i:s', (((int)$value->{'device-created-timestamp'}) / 1000)),
                                'lastSuccessfullBackup'          => (string)$value->{'lastSuccessfullBackup'},
                            );
                            $devices = array_merge(array($device), $devices);
                        }
                    }

                    $response = array(
                        'result' => 'success',
                        'devices' => $devices
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function getFileVersions(){
        if(!empty($_POST['fileName']) && !empty($_POST['filePath'])){
            $headers = array(
                'Content-type'  => 'application/json',
                'Accept'        => 'application/json',
                'Accept-Charset'=> 'utf-8',
                'token'         => $this->session->get('token'),
                'filePath'      => html_entity_decode($_POST['filePath'], ENT_QUOTES | ENT_HTML5),
                'fileName'      => $_POST['fileName'],
                'deviceId'      => $this->session->get('currentDeviceId')
            );
            $result = $this->_model->getFileVersions($headers);
            $this->log('Result:'. json_encode($result), 'info');
            switch ($result['status']) {
                case (200):

                    $fileVersions = [];

                    if($result['body']->{'backupListElement'}->{'fileElementList'}){
                        foreach($result['body']->{'backupListElement'}->{'fileElementList'} as $value){
                            $fileVersion = array(
                                'backupId'            => (string)$value->{'backupId'},
                                'baseFolder'            => (string)$value->{'isBasefolder'},
                                'externalStorage'       => (string)$value->{'isExternalStorage'},
                                'filePath'              => (string)$value->{'filePath'},
                                'fileDevice'            => (string)$value->{'fileDevice'},
                                'fileName'              => (string)$value->{'fileName'},
                                'fileUser'              => (string)$value->{'fileUser'},
                                'folder'                => (string)$value->{'folder'},
                                'clientModifiedTime'    => (!empty($value->{'lastClientModifiedTime'}) && $value->{'lastClientModifiedTime'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'lastClientModifiedTime'}) / 1000)) : '',
                                'serverModifiedTime'    => (!empty($value->{'lastServerModifiedTime'}) && $value->{'lastServerModifiedTime'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'lastServerModifiedTime'}) / 1000)) : '',
                                'checkSum'              => (string)$value->{'checksum'},
                                'exists'                => (string)$value->{'exists'},
                                'rawSize'               => (int)$value->{'size'},
                                'size'                  => $this->getFileSize((string)$value->{'size'}),
                                'version'               => 'Version ',
                                'status'                => (string)$value->{'status'},
                                'storagePlace'          => (string)$value->{'storagePlace'},
                                'fsPath'                => (string)$value->{'fsPath'}
                            );
                            $fileVersions = array_merge(array($fileVersion), $fileVersions);
                        }
                    }

                    $response = array(
                        'result' => 'success',
                        'fileVersions' => array_reverse($fileVersions)
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'post' => $_POST,
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function getFileChunks(){
        if(!empty($_POST['backupId']) && !empty($_POST['userName'])){
            $headers = array(
                'token'         => $this->session->get('token'),
                'backupId'      => $_POST['backupId'],
                'userName'      => $_POST['userName']
            );
            $result = $this->_model->getFileChunks($headers);
            $this->log('Result: '. json_encode($result), 'info');
            switch ($result['status']) {
                case (200):
                    $chunks = [];

                    foreach($result['body']->{'chunkFileList'} as $value){
                        $one = array(
                            'cloudStoragePath'  => (string)$value->{'cloudStoragePath'},
                            'fSPath'            => (string)$value->{'fSPath'},
                            'fileId'            => (string)$value->{'fileId'},
                            'fileName'          => (string)$value->{'fileName'},
                            'fileSource'        => (string)$value->{'fileSource'},
                            'id'                => (string)$value->{'id'},
                            'md5'               => (string)$value->{'md5'},
                            'revision'          => (string)$value->{'revision'},
                            'rawSize'           => (int)$value->{'size'},
                            'size'              => $this->getFileSize(((int)$value->{'size'}/1024)),
                            'uploadedTimeStamp' => (!empty($value->{'uploadedTimeStamp'}) && $value->{'uploadedTimeStamp'} !== '0') ? date('d-M-Y h:i:s', (((int)$value->{'uploadedTimeStamp'}) / 1000)) : '',
                        );
                        $chunks = array_merge(array($one), $chunks);
                    }

                    $response = array(
                        'result' => 'success',
                        'chunks' => $chunks
                    );
                    echo json_encode($response);
                    exit;
                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (445):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('445')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'post' => $_POST,
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function backupFileDelete(){
        if(!empty($_POST['fileName']) && !empty($_POST['filePath']) && !empty($_POST['isFolder'])){

            $fileString = [];

            $headers = array(
                'token'             => $this->session->get('token'),
                'userName'          => $this->session->get('userName'),
                'device'            => $this->session->get('currentDeviceId')
            );

            $result = $this->_model->backupFileDelete($headers, $fileString);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'fileVersions' => $result
                    );

                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'code' => $result,
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'post' => $_POST,
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function getFileSize($sizeInKB){
        $bytes = ($sizeInKB*1024);
        if ($bytes >= 1073741824){
            $bytes = number_format($bytes / 1073741824, 2) . ' GB';
        }elseif ($bytes >= 1048576){
            $bytes = number_format($bytes / 1048576, 2) . ' MB';
        }elseif ($bytes >= 1024){
            $bytes = number_format($bytes / 1024, 2) . ' KB';
        }elseif ($bytes > 1){
            $bytes = $bytes . ' bytes';
        }elseif ($bytes == 1){
            $bytes = $bytes . ' byte';
        }else{
            $bytes = '0 bytes';
        }
        return $bytes;
    }

    public function getFileIcon($ext){
        switch ($ext) {
            case ('zip'):
                $icon = 'file-archive-o';
                break;
            case ('mp3' || 'mp4'):
                $icon = 'file-audio-o';
                break;
            case ('png' || 'jpeg' || 'jpg' || 'gif'):
                $icon = 'file-photo-o';
                break;
            case ('pdf'):
                $icon = 'file-pdf-o';
                break;
            case ('xlsx' || 'xls'):
                $icon = 'file-excel-o';
                break;
            case ('doc' || 'docx'):
                $icon = 'file-word-o';
                break;
            default:
                $icon = 'file';
                break;
        }
        return $icon;
    }

    public function getOsType($deviceType){
        if((strpos($deviceType, 'Wind') === false )){
            if((strpos($deviceType, 'Lin') === false )){
                return 'MAC';
            }else{
                return 'Linux';
            }
        }else{
            return 'Windows';
        }
    }

    public function getOsIcon($deviceType){
        if((strpos($deviceType, 'Wind') === false )){
            if((strpos($deviceType, 'Lin') === false )){
                return 'apple';
            }else{
                return 'linux';
            }
        }else{
            return 'windows';
        }
    }

    public function initiateRestore(){
        if(empty($_POST)){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => 'Path to restore should not be empty'
                )
            );
            exit;
        }
        $newServer = !(empty($_POST['deviceName']) && ($_POST['deviceName'] != 'default')) ? $_POST['deviceName'] : '';
        $path = !(empty($_POST['path'])) ? $_POST['path'] : null;

        $data = array(
            'action'            => 'RESTORE',
            'isDeviceLevel'     => $_POST['deviceLevelRestore'],
            'isRestoreAllVersions'=> $_POST['restoreAllVersions'],
            'deviceUUID'        => !(empty($_POST['deviceUUID'])) ? $_POST['deviceUUID'] : $this->session->get('currentDeviceUUID'),
            'paths'             => $_POST['paths'],
            'destinationPath'   => $path
        );

        $hostName = !empty($newServer) ? $newServer : $this->session->get('currentDeviceName');

        $this->log('Data'. json_encode($data), 'info');
        $this->log('ServerName'. $hostName, 'info');

        $context = new ZMQContext();
        $client = $this->client_socket($context, $hostName);
        $sequence = 0;
        $retries_left = 3;
        $read = $write = array();
        while ($retries_left) {
            $client->send(json_encode($data));
            $expect_reply = true;
            while ($expect_reply) {
                $poll = new ZMQPoll();
                $poll->add($client, ZMQ::POLL_IN);
                $events = $poll->poll($read, $write, 30000);
                --$retries_left;
                if ($events > 0) {

                    $result = $client->recv();
                    $this->log('Result'. json_encode($result), 'info');
                    switch ($result) {
                        case (200):

                            $response = array(
                                'result' => 'success',
                                'post' => $_POST,
                                'msg' => $this->language->get('restoreInitiated')
                            );
                            echo json_encode($response);
                            exit;

                        case (421):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('421')
                            );
                            echo json_encode($result);
                            exit();

                        case (600):
                            $result = array(
                                'result' => 'fail',
                                'type' => 'restore',
                                'msg' => $this->language->get('600')
                            );
                            echo json_encode($result);
                            exit();

                        case (601):
                            $result = array(
                                'result' => 'fail',
                                'type' => 'busy',
                                'msg' => $this->language->get('601')
                            );
                            echo json_encode($result);
                            exit();

                        case (602):
                            $result = array(
                                'result' => 'fail',
                                'type' => 'busy',
                                'msg' => $this->language->get('602')
                            );
                            echo json_encode($result);
                            exit();

                        case (500):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('500')
                            );
                            echo json_encode($result);
                            exit();
                        case (0):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('0')
                            );
                            echo json_encode($result);
                            exit();
                        default:
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('default')
                            );
                            echo json_encode($result);
                            exit();
                    }
                } elseif ($retries_left == 0) {
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    break;
                } else {
                    $this->log('Retrying to connect to client:'.$hostName, 'info');
                    $client = $this->client_socket($context, $hostName);
                    $client->send($sequence);
                }
            }
        }
    }

    public function checkRestoreProgress(){
        if(empty($_POST)){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
            exit;
        }
        $newServer = !(empty($_POST['deviceName'])) ? $_POST['deviceName'] : '';
        $data = array(
            'action' => 'ISRESTORERUNNING'
        );
        $hostName = !empty($newServer) ? $newServer : $this->session->get('currentDeviceName');

        $this->log('Restore'. json_encode($hostName), 'info');

        $context = new ZMQContext();
        $client = $this->client_socket($context, $hostName);

        $sequence = 0;
        $retries_left = 3;
        $read = $write = array();

        while ($retries_left) {
            $client->send(json_encode($data));
            $expect_reply = true;
            while ($expect_reply) {

                $poll = new ZMQPoll();
                $poll->add($client, ZMQ::POLL_IN);
                $events = $poll->poll($read, $write, 3000);
                --$retries_left;
                if ($events > 0) {

                    $result = $client->recv();

                    switch ($result) {
                        case (200):

                            $response = array(
                                'result' => 'success',
                                'msg' => $this->language->get('restoreInProgress')
                            );
                            echo json_encode($response);
                            exit;

                        case (420):
                            $result = array(
                                'result' => 'fail',
                                'type' => 'restore',
                                'msg' => $this->language->get('420')
                            );
                            echo json_encode($result);
                            exit();

                        case (425):
                            $result = array(
                                'result' => 'fail',
                                'type' => 'busy',
                                'msg' => $this->language->get('425')
                            );
                            echo json_encode($result);
                            exit();

                        case (500):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('500')
                            );
                            echo json_encode($result);
                            exit();
                        case (0):
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('0')
                            );
                            echo json_encode($result);
                            exit();
                        default:
                            $result = array(
                                'result' => 'fail',
                                'msg' => $this->language->get('default')
                            );
                            echo json_encode($result);
                            exit();
                    }
                } elseif ($retries_left == 0) {
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    break;
                } else {
                    $result = array(
                        'result' => 'fail',
                        'msg' => 'Re-Trying to connect to the server please wait'
                    );

                    $client = $this->client_socket($context, $_POST['serverHostName']);
                    $client->send($sequence);
                }
            }
        }
    }

    public function client_socket(ZMQContext $context, $hostName){
        $client = new ZMQSocket($context, ZMQ::SOCKET_REQ);
        try{
            $client->connect("tcp://".$hostName.":5555");
            $client->setSockOpt(ZMQ::SOCKOPT_LINGER, 0);
            return $client;
        }catch (\Exception $e){
            $this->log('Exception: '. $e, 'info');
            $result = array(
                'result' => 'fail',
                'message' => 'Could not connect to the Server'
            );
            echo json_encode($result);
            exit();
        }
    }

    public function getAllUnblockedDevices(){
        if(empty($_GET)){
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData'),
                    'redirect' => (!empty($this->session->get('currentDeviceUUID'))) ? true : false
                )
            );
            exit;
        }

        $headers = array(
            'Content-Type' => 'application/json',
            'token' => $this->session->get('token'),
            'skipValue' => $_GET['skipValue']
        );
        $isCurrentDeviceBlocked = false;
        if(!empty($_GET['q']) && $_GET['q'] != 'all'){
            $headers['deviceNames'] = $_GET['q'];
            $result = $this->_model->searchInDevices($headers);
        }else{
            $result = $this->_model->getAllDevices($headers);
        }

        switch ($result['status']) {
            case (200):

                $devices = [];

                if(empty($result['body'])){
                    $response = array(
                        'result' => 'success',
                        'devices' => $devices
                    );
                    echo json_encode($response);
                    exit;
                }

                foreach($result['body']->{'device-info'} as $value){
                    $device = array(
                        'profilePic' => Url::profilePic((string)$value->{'userName'}),
                        'decoupled' => (string)$value->{'decoupled'},
                        'createdTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'device-created-timestamp'}) / 1000)),
                        'modifiedTimeStamp' => date('d-m-Y h:i:s', (((int)$value->{'modified-timestamp'}) / 1000)),
                        'modifiedTimeStampUnix' => (int)$value->{'modified-timestamp'},
                        'deviceName' => (string)$value->{'device-name'},
                        'osType' => (string)$value->{'device-os-type'},
                        'icon' => (strpos((string)$value->{'device-os-type'},'Win') !== false ? 'windows' : 'apple'),
                        'version' => (string)$value->{'device-os-version'},
                        'tempPath' => (string)$value->{'device-temp-path'},
                        'deviceUUID' => (string)$value->{'device-uuid'},
                        'blocked' => (string)$value->{'is-blocked'},
                        'userName' => (string)$value->{'userName'},
                    );
                    if($device['blocked'] == 'false'){
                        $devices = array_merge(array($device), $devices);
                    }else{
                        if($device['deviceName'] == $this->session->get('currentDeviceName')){
                            $isCurrentDeviceBlocked = true;
                        }
                    }
                }

                if($isCurrentDeviceBlocked == false){
                    $devices = array_merge(array(array('deviceName' => 'All', 'deviceUUID' => 'all', 'profilePic' => Url::profilePic('default'))), $devices);
                }
                $response = array(
                    'result' => 'success',
                    'devices' => $devices
                );
                echo json_encode($response);
                exit;
            case (421):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('421')
                );
                echo json_encode($result);
                exit();
            case (500):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('500')
                );
                echo json_encode($result);
                exit();
            case (0):
                $result = array(
                    'result' => 'fail',
                    'msg' => $this->language->get('0')
                );
                echo json_encode($result);
                exit();
            default:
                $result = array(
                    'result' => 'fail',
                    'code' => $result,
                    'msg' => $this->language->get('default')
                );
                echo json_encode($result);
                exit();
        }
    }

    public function shareFile(){

        if(!empty($_POST)){
            $headers = array(
                'Content-Type'      => 'application/json',
                'forceShare'        => 'true',
                'token'             => $this->session->get('token'),
                'userName'          => $this->session->get('userName'),
                'productType'       => "backup",
                'backupId' => $_POST['backupId'],
            );

            $shareElements = array(
                'userName' => $this->session->get('userName'),
                'cloudName' => $this->session->get('cloudName'),
                'miniCloudName' => '',
                'devicePath' => $_POST['filePath'],
                'fileName' => $_POST['fileName'],
                'passwordProtected' => (string)$_POST['isPasswordProtected'],
                'passwordHash' => !empty($_POST['isPasswordProtected'] == true) ? md5($_POST['password']) : '',
                'validUntil' =>  !empty($_POST['validTill']) ? (strtotime($_POST['validTill']." +1 day")*1000) : 0,
                'download' => (string)$_POST['download'],
                'view' => (string)$_POST['view']
            );
            $this->log(json_encode($shareElements), "info");
            $result = $this->_model->shareFile($headers, $shareElements);

            switch ($result['status']) {
                case (200):
                    $shared = $result['body']->{'sharedFileImageElement'};
                    if($shared->{'passwordProtected'} == true){
                        $type = md5('protected');
                    }else{
                        $type = md5('read');
                    }
                    $link = SHARE_PATH.$type.DS.$shared->{'key12'}.DS.$shared->{'key14'}.DS.$shared->{'key18'}.DS.$shared->{'key20'};

                    $response = array(
                        'result' => 'success',
                        'link' => $link,
                        'data' => $result['body']->{'sharedFileImageElement'}
                    );
                    echo json_encode($response);
                    exit;
                case (445):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('445')
                    );
                    echo json_encode($result);
                    exit();

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function updateSharePolicy(){

        if(!empty($_POST)){
            $headers = array(
                'Content-Type'      => 'application/json',
                'forceShare'        => 'true',
                'token'             => $this->session->get('token'),
                'userName'          => $this->session->get('userName'),
                'productType'       => "backup",
                'backupId' => $_POST['backupId'],
            );

            list($key12, $key14, $key18, $key20) = explode('+', $_POST['key']);
            $shareElements		= array(
                'userName' => $this->session->get('userName'),
                'cloudName' => $this->session->get('cloudName'),
                'miniCloudName' => '',
                'devicePath' => htmlspecialchars_decode($_POST['filePath']),
                'fileName' => $_POST['fileName'],
                'passwordProtected' => (string)$_POST['isPasswordProtected'],
                'passwordHash' => !empty($_POST['isPasswordProtected'] == true) ? md5($_POST['password']) : '',
                'validUntil' =>  (strtotime($_POST['validTill']." +1 day")*1000),
                'download' => (string)$_POST['download'],
                'key12' => $key12,
                'key14' => $key14,
                'key18' => $key18,
                'key20' => $key20,
                'view' => (string)$_POST['view']
            );

            $result = $this->_model->updateSharePolicy($headers, $shareElements);

            switch ($result['status']) {
                case (200):
                    $shared = $result['body']->{'sharedFileImageElement'};
                    if($shared->{'passwordProtected'} == true){
                        $type = md5('protected');
                    }else{
                        $type = md5('read');
                    }
                    $link = SHARE_PATH.$type.DS.$shared->{'key12'}.DS.$shared->{'key14'}.DS.$shared->{'key18'}.DS.$shared->{'key20'};

                    $response = array(
                        'result' => 'success',
                        'link' => $link,
                        'data' => $result['body']->{'sharedFileImageElement'}
                    );
                    echo json_encode($response);
                    exit;

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (445):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('445')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function unShare(){
        if(!empty($_POST)){
            $headers = array(
                'Content-Type' => 'application/xml',
                'Accept' => 'application/xml',
                'Accept-Charset' => 'utf-8',
                'token'             => $this->session->get('token'),
                'userName'          => $this->session->get('userName'),
                'backupId'          => $_POST['backupID'],
                'productType'       => "backup"
            );
            $xml ='<share-info>
				<shared-time><![CDATA['.time().']]></shared-time>
				<user-name><![CDATA['.$this->session->get('userName').']]></user-name>
				<cloud-name><![CDATA['.$this->session->get('cloudName').']]></cloud-name>
				<mini-cloud-name><![CDATA[]]></mini-cloud-name>
				<device-path><![CDATA['.htmlspecialchars_decode($_POST['filePath']).']]></device-path>
				<file-name><![CDATA['.$_POST['fileName'].']]></file-name>
				<is-password-protected>false</is-password-protected>
				<password-hash></password-hash>
			</share-info>';
            $this->log(json_encode($headers), "info");
            $result = $this->_model->unShare($headers, $xml);
            switch ($result['status']) {
                case (200):

                    $response = array(
                        'result' => 'success',
                        'msg' => $this->language->get('fileUnshared'),
                        'data' => $result
                    );
                    echo json_encode($response);
                    exit;

                case (204):

                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('fileWasNotShared')
                    );
                    echo json_encode($result);
                    exit();

                case (421):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('421')
                    );
                    echo json_encode($result);
                    exit();
                case (445):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('445')
                    );
                    echo json_encode($result);
                    exit();
                case (500):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('500')
                    );
                    echo json_encode($result);
                    exit();
                case (0):
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('0')
                    );
                    echo json_encode($result);
                    exit();
                default:
                    $result = array(
                        'result' => 'fail',
                        'msg' => $this->language->get('default')
                    );
                    echo json_encode($result);
                    exit();
            }

        }else{
            echo json_encode(array(
                    'result' => 'noData',
                    'message' => $this->language->get('noData')
                )
            );
        }
    }

    public function openFileLocation(){
        if($_GET['filePath']){
            $this->session->put('filePath', htmlentities($_GET['filePath']));
            Url::redirect('backup/files/'.htmlentities($_GET['deviceID']).'/'.htmlentities($_GET['deviceName']).'/'.htmlentities($_GET['deviceUUID']).'');
        }else{
            Url::redirect('404');
        }
    }

    public function collectStats(){

        $browser = array(
            'version'   => '0.0.0',
            'majorver'  => 0,
            'minorver'  => 0,
            'build'     => 0,
            'name'      => 'unknown',
            'useragent' => ''
        );

        $browsers = array(
            'firefox', 'msie', 'opera', 'chrome', 'safari', 'mozilla', 'seamonkey', 'konqueror', 'netscape',
            'gecko', 'navigator', 'mosaic', 'lynx', 'amaya', 'omniweb', 'avant', 'camino', 'flock', 'aol'
        );

        if (isset($_SERVER['HTTP_USER_AGENT'])) {
            $browser['useragent'] = $_SERVER['HTTP_USER_AGENT'];
            $user_agent = strtolower($browser['useragent']);
            foreach($browsers as $_browser) {
                if (preg_match("/($_browser)[\/ ]?([0-9.]*)/", $user_agent, $match)) {
                    $browser['name'] = $match[1];
                    $browser['version'] = $match[2];
                    @list($browser['majorver'], $browser['minorver'], $browser['build']) = explode('.', $browser['version']);
                    break;
                }
            }
        }
        $browserName = $browser['name'] .' - '. $browser['version'];
        if (getenv('HTTP_CLIENT_IP')){
            $ipAddress = getenv('HTTP_CLIENT_IP');
        }
        else if(getenv('HTTP_X_FORWARDED_FOR')) {
            $ipAddress = getenv('HTTP_X_FORWARDED_FOR');
        }
        else if(getenv('HTTP_X_FORWARDED')){
            $ipAddress = getenv('HTTP_X_FORWARDED');
        }
        else if(getenv('HTTP_FORWARDED_FOR')){
            $ipAddress = getenv('HTTP_FORWARDED_FOR');
        }
        else if(getenv('HTTP_FORWARDED')){
            $ipAddress = getenv('HTTP_FORWARDED');
        }
        else{
            $ipAddress = 'UNKNOWN';
        }
        $remote 	= $_SERVER['REMOTE_ADDR'] ;
        $language 	= $_SERVER['HTTP_ACCEPT_LANGUAGE'];
        return array("ip" => $ipAddress,"remoteAdd" => $remote,
            "browser" => $browserName,"language" => $language);
    }

    public function getOS($user_agent) {

        $os_platform    =   "Unknown OS Platform";
        $os_array       =   array(
            '/windows nt 10/i'     =>  'Windows 10',
            '/windows nt 6.3/i'     =>  'Windows 8.1',
            '/windows nt 6.2/i'     =>  'Windows 8',
            '/windows nt 6.1/i'     =>  'Windows 7',
            '/windows nt 6.0/i'     =>  'Windows Vista',
            '/windows nt 5.2/i'     =>  'Windows Server 2003/XP x64',
            '/windows nt 5.1/i'     =>  'Windows XP',
            '/windows xp/i'         =>  'Windows XP',
            '/windows nt 5.0/i'     =>  'Windows 2000',
            '/windows me/i'         =>  'Windows ME',
            '/win98/i'              =>  'Windows 98',
            '/win95/i'              =>  'Windows 95',
            '/win16/i'              =>  'Windows 3.11',
            '/macintosh|mac os x/i' =>  'Mac OS X',
            '/mac_powerpc/i'        =>  'Mac OS 9',
            '/linux/i'              =>  'Linux',
            '/ubuntu/i'             =>  'Ubuntu',
            '/iphone/i'             =>  'iPhone',
            '/ipod/i'               =>  'iPod',
            '/ipad/i'               =>  'iPad',
            '/android/i'            =>  'Android',
            '/blackberry/i'         =>  'BlackBerry',
            '/webos/i'              =>  'Mobile'
        );
        foreach ($os_array as $regex => $value) {
            if (preg_match($regex, $user_agent)) {
                $os_platform    =   $value;
            }
        }
        return $os_platform;
    }
}
